/* setup.c */ 

#include <stdio.h> 
#include <stdlib.h> 

#include "consts.h" 
#include "setup.h" 

/*----------------------------------------------------------------------*/  
/* Function Defined */ 

void SetUpDisplay( int rows,
                   int cols,
                   int *radius,
                   int *interval,
                   int *border,
                   int *shiftX,
                   int *shiftY,
                   int *winW,
                   int *winH      )
{
	long wd, ht; 
	int status = VALID; 

	(*radius)   = RADIUS; 
	(*interval) = INTERVAL; 
	(*border)   = BORDER; 
	wd = CalcLengthI( (*radius), (*interval), cols, (*border) ); 
	ht = CalcLengthI( (*radius), (*interval), rows, (*border) ); 

	if( wd > MAXWIDTH || ht > MAXHEIGHT ) 
	{ 
		ResetIndependents( radius, 
				   interval, 
				   border, 
				   rows, 
				   cols, 
				   &wd, 
				   &ht       ); 
	} 

	if( (*radius) >= MINRAD ) 
	{ 
		status = SetDependents( wd, 
					ht, 
					shiftX, 
					shiftY, 
					winW, 
					winH    ); 
	} 
	else 
	{ 
		(*interval) = INTERVAL/2; 
		wd = CalcLengthF( PTWIDTH, (*interval), cols, (*border) ); 
		ht = CalcLengthF( PTWIDTH, (*interval), rows, (*border) ); 
		SetDependents( wd, ht, shiftX, shiftY, winW, winH ); 
	} 

	if( status == INVALID ) 
	{ 
		fprintf( stderr, "\nError occurred!" ); 
		fprintf( stderr, "\nAborting!\n" ); 
		fflush( stderr ); 
		MPI_Abort( MPI_COMM_WORLD, 1 ); 
		exit( 1 ); 
	} 
} 

/*----------------------------------------------------------------------*/ 
int SetDependents( long w, long h, int *sX, int *sY, int *wW, int *wH ) 
{ 
	if( w <= DEFWIDTH )
	{ 
		(*sX) = (int)( ( DEFWIDTH - w )/2 ); 
		(*wW) = DEFWIDTH; 
	} 
	else if( w <= MAXWIDTH ) 
	{ 
		(*sX) = 0; 
		(*wW) = (int)w; 
	} 
	else 
	{ 
		(*sX) = 0; 
		(*wW) = (int)w; 
		return INVALID; 
	} 

	if( h <= DEFHEIGHT ) 
	{ 
		(*sY) = (int)( ( DEFHEIGHT - h)/2 );
		(*wH) = DEFHEIGHT; 
	} 
	else if( h <= MAXHEIGHT ) 
	{ 
		(*sY) = 0; 
		(*wH) = h; 
	} 
	else 
	{ 
		(*sY) = 0; 
		(*wH) = h; 
		return INVALID; 
	} 

	return VALID; 
} 

/*----------------------------------------------------------------------*/ 
void ResetIndependents( int  *r, 
			int  *i, 
			int  *b, 
			int  rows, 
			int  cols, 
			long *wd, 
			long *ht       ) 
{ 
	do 
	{ 
		(*r)--; 
		if( (*r) <= ( RADIUS/2 ) ) 
		{ 
			(*b) = BORDER/2; 
			if( (*r) <= ( INTERVAL + 1 ) ) 
				(*i) = INTERVAL/2; 
			else if( (*r) <= 0 ) 
			{ 
				(*r) = 0; 
				(*i) = 0; 
			} 
		} 
		(*wd) = CalcLengthI( (*r), (*i), cols, (*b) ); 
		(*ht) = CalcLengthI( (*r), (*i), rows, (*b) ); 
	} 
	while( (*wd) > MAXWIDTH || (*ht) > MAXHEIGHT ); 
} 

/*----------------------------------------------------------------------*/ 
long CalcLengthI( int r, int i, int n, int b ) 
{ 
	return (long)( ( 2*r + i )*n - i + 2*b ); 
} 

/*----------------------------------------------------------------------*/ 
long CalcLengthF( float r, int i, int n, int b ) 
{ 
	long  tmp; 
	float size; 

	size = ( 2*r + (float)i )*(float)n - (float)i + 2*(float)b; 
	tmp = (long)size; 
	if( size > (float)tmp ) 
		size = size + 1.0; 

	return (long)size; 
} 

/*----------------------------------------------------------------------*/ 
int InitGraph( MPE_XGraph *graph,
               int my_rank,
               int x,
               int y,
               int w,
               int h )
{
        int result;
        int my_window_opened;
        int all_windows_opened;

        result = MPE_Open_graphics( graph,
                                    MPI_COMM_WORLD,
                                    (char *)0,
                                    x,
                                    y,
                                    w,
                                    h,
                                    0 );
        if( result == MPI_SUCCESS )
                my_window_opened = 1;
        MPI_Allreduce( &my_window_opened,
                       &all_windows_opened,
                       1,
                       MPI_INT,
                       MPI_LAND,
                       MPI_COMM_WORLD );

        if( !all_windows_opened )
        {
                if( my_rank == 0 )
                {
                        fprintf( stderr, "\nNot all windows opened!" );
                        fprintf( stderr, "\nAborting!\n" );
                        fflush( stderr );
                }
                if( my_window_opened )
                        MPE_Close_graphics( graph );
                MPI_Abort( MPI_COMM_WORLD, 1 );

                return 0;
        }

        return 1;
}

/* -------------------------------------------------------------- */
 
