/*
 * comm.c 
 *
 * Communication functions
 */

#include <stdio.h>
#include <string.h>
#include "mpi.h"
#include "comm.h"
#include "utility.h"

/*================================================================*/
/* Plug-in replacement for MPI_Allgather.  Can be used if synaptic
 * connections are only between neurons belonging to adjacent 
 * processes:  i.e., if neuron i is assigned process q, then neuron
 * i sends synaptic outputs only to neurons assigned to processes
 * q-1, q, and q+1.  
 *
 * Notes:  
 *     1.  In order to use this function, set the macro
 *         ALL_GATHER to Pairwise_exchange in equations.h:
 *         i.e.,
 *         #define ALL_GATHER Pairwise_exchange
 *     2.  The current definition does _not_ exchange "round
 *         the corner -- i.e., process 0 will exchange only with
 *         process 1, and process num_processes - 1 will exchange
 *         only with process num_processes - 2.
 *     3.  Assumes that the function is always called with the
 *         same communicator.
 *     4.  Assumes the arrays sendbuf and recvbuf are both arrays
 *         of doubles.
 *     5.  Assumes recvbuf contains enough storage for synaptic
 *         outputs of _all_ neurons although only the storage
 *         corresponding to processes q-1, q, and q+1 will be
 *         used.
 *     6.  Assumes sendcount = recvcount, and all calls use the
 *         same values for sendcount.
 *     7.  Doesn't work with discontiguous buffers.
 */
int Pairwise_exchange (
        double*           sendbuf     /* IN  */,
        int               sendcount   /* IN  */,
        MPI_Datatype      sendtype    /* IN  */,
        double*           recvbuf     /* OUT */,
        int               recvcount   /* IN  */,
        MPI_Datatype      recvtype    /* IN  */,
        MPI_Comm          comm        /* OUT */) {

    static bool  first_time = true;
    static int   my_rank;
    static int   num_processes;
    static int   right_nbhr;
    static int   left_nbhr;
    static int   my_offset;
    static int   right_offset;
    static int   left_offset;

    MPI_Request  right_send_request;
    MPI_Status   right_send_status;
    MPI_Request  requests[3];
    MPI_Status   statuses[3];

    if (first_time) {
        MPI_Comm_rank(comm, &my_rank);
        MPI_Comm_size(comm, &num_processes);
        my_offset = my_rank*recvcount;
        right_nbhr = my_rank + 1;
        right_offset = right_nbhr*recvcount;
        left_nbhr = my_rank - 1;
        left_offset = left_nbhr*recvcount;
        if (left_nbhr < 0) {
            left_nbhr = MPI_PROC_NULL;
            left_offset = 0;
        }
        if (right_nbhr >= num_processes) {
            right_nbhr = MPI_PROC_NULL;
            right_offset = 0;
        }
        first_time = false;
    }

    memcpy(recvbuf + my_offset, sendbuf, sendcount*sizeof(double));

    if (num_processes == 1) {
        return MPI_SUCCESS;
    }
    
    /* Tag 0 for right send, Tag 1 for left send */
    MPI_Isend(sendbuf, sendcount, sendtype, right_nbhr, 0, comm,
        &right_send_request);
    MPI_Irecv(recvbuf + left_offset, recvcount, recvtype,
         left_nbhr, 0, comm, &(requests[0]));
    MPI_Wait(&right_send_request, &right_send_status);

    MPI_Isend(sendbuf, sendcount, sendtype, left_nbhr, 1, comm,
        &(requests[1]));
    MPI_Irecv(recvbuf + right_offset, recvcount, recvtype,
         right_nbhr, 1, comm, &(requests[2]));
    MPI_Waitall(3, requests, statuses);
    
    return MPI_SUCCESS;
}  /* Pairwise_exchange */

