/* File:     create_read_constants.c
 * Purpose:  Create a C source file that can be used by Neurosys in
 *           reading in the constants describing the models.  The
 *           C source file creates an MPI datatype for broadcasting
 *           the constants, and then reads and broadcasts the constants.
 * Input:    A template file "read_constants.t"
 *           A file containing a list of the constant names called 
 *               constant_list.  The names of the int constants should
 *               be listed first.
 * Output:   read_constants.c
 *
 * Note:     This program should be run *before* making the solver.
 */
#include <stdio.h>
#include <stdlib.h>
#include "equations.h"  /* Has #defines for INT_COUNT, DOUBLE_COUNT */

#define MAX_NAME 80
#define MARKER '$'    /* Indicates text is to be added to template */

void Check_file_open(FILE* fp, char* name);

void Skip_comment(FILE* template);
void Add_first_value(FILE* template, FILE* c_file, 
        FILE* constant_list);
void Add_offset_calcs(FILE* template, FILE* c_file, 
        FILE* constant_list);
void Add_nscanfs(FILE* template, FILE* c_file, FILE* constant_list);
void Add_bcast(FILE* template, FILE* c_file, FILE* constant_list);
void Add_debug_statements(FILE* template, FILE* c_file, 
         FILE* constant_list);
void Finish_read_and_print(FILE* template, FILE* c_file);

void Read_and_print_to_marker(FILE* template, FILE* c_file);

int main(void) {
    FILE* template;
    FILE* c_file;
    FILE* constant_list;

    template = fopen("read_constants.t", "r");
    Check_file_open(template, "read_constants.t");
    c_file = fopen("read_constants.c", "w");
    Check_file_open(c_file, "read_constants.c");
    constant_list = fopen("constant_list", "r");
    Check_file_open(constant_list, "constant_list");

    Skip_comment(template);
    Add_first_value(template, c_file, constant_list);
    Add_offset_calcs(template, c_file, constant_list);
    Add_nscanfs(template, c_file, constant_list);
    Add_bcast(template, c_file, constant_list);
    Add_debug_statements(template, c_file, constant_list);
    Finish_read_and_print(template, c_file);

    fclose(template);
    fclose(c_file);
    fclose(constant_list);

    return 0;
}  /* main */


/*------------------------------------------------------------------*/
void Check_file_open(FILE* fp, char* name) {
    if (fp == (FILE*) NULL) {
        fprintf(stderr, "Can't open %s\n", name);
        exit(-1);
    }
}  /* Check_file_open */


/*------------------------------------------------------------------*/
void Skip_comment(FILE* template) {
    char c1, c2;
    int done = 0;

    fscanf(template, "%c", &c1);
    while (!done) {
        if (c1 == '*') {
            fscanf(template, "%c", &c2);
            if (c2 == '/')
                done = 1;
            else
                c1 = c2;
        } else {
            fscanf(template, "%c", &c1);
        }
    }  /* while */
    
    /* Skip newline */
    fscanf(template, "%c", &c1);
}  /* Skip_comment */


/*------------------------------------------------------------------*/
void Add_first_value(FILE* template, FILE* c_file, 
        FILE* constant_list) {
    char first_const[MAX_NAME];

    Read_and_print_to_marker(template, c_file);
    fscanf(constant_list, " %s", first_const);
    fprintf(c_file, "%s", first_const);
}  /* Add_first_three_values */


/*------------------------------------------------------------------*/
void Add_offset_calcs(FILE* template, FILE* c_file, 
        FILE* constant_list) {
    char constant[MAX_NAME];
    int total = INT_COUNT + DOUBLE_COUNT;
    int i;

    Read_and_print_to_marker(template, c_file);
    for (i = 1; i < total; i++) {
        fscanf(constant_list, " %s", constant);
        fprintf(c_file, "    MPI_Address(&%s, &address);\n", constant);
        fprintf(c_file, "    displacements[%d] = address - ", i);
        fprintf(c_file, "start_address;\n\n");
    }
}  /* Add_offset_calcs */


/*------------------------------------------------------------------*/
void Add_nscanfs(FILE* template, FILE* c_file, FILE* constant_list) {
    int i;
    char constant[MAX_NAME];

    rewind(constant_list);
    Read_and_print_to_marker(template, c_file);
   
    for (i = 0; i < INT_COUNT; i++) {
        fscanf(constant_list, " %s", constant);
        fprintf(c_file, "        nscanf(fp, \"%%d\", &%s);\n", 
            constant);
    }
    for (i = 0; i < DOUBLE_COUNT; i++) {
        fscanf(constant_list, " %s", constant);
        fprintf(c_file, "        nscanf(fp, \"%%lf\", &%s);\n", 
            constant);
    }
}  /* Add_nscanfs */


/*------------------------------------------------------------------*/
void Add_bcast(FILE* template, FILE* c_file, FILE* constant_list) {
    char constant[MAX_NAME];

    rewind(constant_list);
    Read_and_print_to_marker(template, c_file);
    fscanf(constant_list, " %s", constant);
    fprintf(c_file, "    MPI_Bcast(&%s, 1, ", constant);
    fprintf(c_file, "constants_mpi_t, 0, MPI_COMM_WORLD);");
}  /* Add_bcast */


/*------------------------------------------------------------------*/
void Add_debug_statements(FILE* template, FILE* c_file, 
         FILE* constant_list) {
    char constant[MAX_NAME];
    int i;

    rewind(constant_list);
    Read_and_print_to_marker(template, c_file);
  
    for (i = 0; i < INT_COUNT; i++) {
        fscanf(constant_list, " %s", constant);
        fprintf(c_file, "    printf(\"Process %%d > %s = %%d\\n\", ",
            constant);
        fprintf(c_file, "my_rank, %s);\n", constant);
    }
    for (i = 0; i < DOUBLE_COUNT; i++) {
        fscanf(constant_list, " %s", constant);
        fprintf(c_file, "    printf(\"Process %%d > %s = %%f\\n\", ",
            constant);
        fprintf(c_file, "my_rank, %s);\n", constant);
    }
}  /* Add_debug_statements */


/*------------------------------------------------------------------*/
void Finish_read_and_print(FILE* template, FILE* c_file) {
    char c;
    int ret_val;

    ret_val = fscanf(template, "%c", &c);
    while (ret_val != EOF) {
        fprintf(c_file, "%c", c);
        ret_val = fscanf(template, "%c", &c);
    }
}  /* Finish_read_and_print */


/*------------------------------------------------------------------*/
void Read_and_print_to_marker(FILE* template, FILE* c_file) {
    char c;

    fscanf(template, "%c", &c);
    while (c != MARKER) {
        fprintf(c_file, "%c", c);
        fscanf(template, "%c", &c);
    }
    
}  /* Read_and_print_to_marker */


