/*
 * main.c
 *
 * Purpose:  Use MPI and a parallel ODE solver to solve a system 
 *     of ordinary differential equations modelling a network of 
 *     biological neurons
 *
 * Input:  
 *    name of file containing constants used in models   
 *    file containing constants
 *    'd' or 's' for input from a data file or stdin
 *    if 'd', name of file containing input data, and file
 *    if 's', 
 *        initial time,
 *        step size,
 *        number of time steps, 
 *        number of variables dimension of system (including synaptic
 *            variables),
 *        number of neurons, 
 *        number of rows of neurons (currently unused),
 *        number of columns of neurons (currently unused),
 *        how often calculations are printed to output (an int),
 *        output filename,
 *        array of model types (ints in range 0--MODEL_TYPE_COUNT-1),
 *        initial conditions,
 *        array containing number of synaptic inputs to each neuron,
 *        adjacency list
 *
 * Output: in output file -- membrane voltages at each time step, and,
 *        if undistributed output, time
 *    Prompts for input.
 *    Debug output.
 *
 * Notes:
 *    1.  Number of neurons must be evenly divisible by the number
 *        of processes.
 *    2.  Hence each process is assigned the same number of neurons.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "mpi.h"

/* For debugging only */
#include <unistd.h>

#include "equations.h"
#include "read_constants.h"
#include "output.h"
#include "input.h"
#include "solver.h"
#include "utility.h"
#include "nscanf.h"
#include "globals.h"


/*===================================================================*/
int main(int argc, char* argv[]) {
    /* Variables needed by MPI */
    char  hostname[MPI_MAX_PROCESSOR_NAME];
    int   name_len;

    /* Variables needed by solver */
    double*      y_init;
    double*      syn_init;

    int          m;
    double       start_time;
    double       stop_time;

    MPI_Init(&argc,&argv);

    comm = MPI_COMM_WORLD;
    MPI_Comm_size(comm, &num_processes);
    MPI_Comm_rank(comm, &my_rank);

/*  #   ifdef DEBUG */
    /* The printf, hostname, and barrier  are for debugging only */
    MPI_Get_processor_name(hostname, &name_len);
    printf("Process %d > Pid = %d running on %s \n", my_rank, 
           getpid(), hostname);
    fflush(stdout);

    /* Force all processes to block */
/*
    if (my_rank == 0) {
        int i;
        printf("Please enter an int\n");
        nscanf(stdin, "%d", &i);
    }
    MPI_Barrier(comm);
*/
/*  #   endif */

    /* Initialize Equation Constants defined in equations.c */
    Read_constants();

    /* Get other input data */
    Read_data(&y_init, &syn_init);

#   ifdef DEBUG
    printf("Process %d > After read_data.\n", my_rank);
    fflush(stdout);
#   endif 

#   ifdef DIST_OUTPUT
        Setup_output();
#   else
        if (my_rank == 0) Setup_output();
#   endif  /* DIST_OUTPUT */
       
    start_time = MPI_Wtime();
    DE_SOLVE(y_init, syn_init);
    stop_time = MPI_Wtime();

#   ifdef DEBUG
    printf("Process %d > After DE_SOLVE.\n", my_rank);
    fflush(stdout);
#   endif


#   ifdef DIST_OUTPUT
        Print_output_buffer();
        Close_output();
        if (my_rank == 0)
           printf("Elapsed time for solve = %f seconds\n", 
                   stop_time - start_time);
#   else
        if (my_rank == 0) {
           Print_output_buffer();
           Close_output();
           printf("Elapsed time for solve = %f seconds\n", 
                   stop_time - start_time);
        }
#   endif  /* DIST_OUTPUT */

    free(neuron_types);
    free(y_init);
    free(syn_init);
    for(m = 0; m < num_my_neurons; m++) {
        if(((syn_connections)[m]).dimension > 0) {
          if (syn_connections[m].links != (int*) NULL)
              free(((syn_connections)[m]).links);
          if (syn_connections[m].signs != (int*) NULL)
              free(syn_connections[m].signs);
        }
    }
    free(syn_connections);

    Cleanup_utilities();

    MPI_Finalize();

    return 0;
}  /* main */
