/*
 * nscanf.c
 *
 * Replacement for fscanf that ignores comments.  
 *     A comment is any text preceded by a # and terminated by
 *     a newline.
 * Current version can handle %d (decimal int), %s (string), 
 *     %f (float), %lf (double) and " %c" -- i.e., a non-
 *     whitespace char.
 * Any input token (int, string, etc.) must be separated from 
 *     following token by whitespace (space, tab, newline).
 */
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <stdlib.h>
#include "nscanf.h"

#define COMMENT_CHAR '#'

/* Private functions */
void Go_to_end_of_line(FILE* fp  /* IN/OUT */);

void Get_buf(
           FILE*   fp        /* IN/OUT */, 
           char*   buf       /* OUT    */);

int Whitespace(char c  /* IN */);

char* Advance(char* buf_p /* IN */);

void nscanf(
         FILE*  fp      /* IN */, 
         char*  format  /* IN */, 
         ...) {

    char buf[BUFSIZ];
    char    *fmt_p;
    va_list arg_p;
    char    *sval_p;
    int     *ival_p;
    char    *cval_p;
    float   *fval_p;
    double  *dval_p;
    
    fmt_p = format;
    va_start(arg_p, format);  /* arg_p points to first unnamed arg */

    while (*fmt_p != '\0') {

        if (*fmt_p == '%') {
            fmt_p++;
            switch (*fmt_p) {
                case 'd':
                    ival_p = va_arg(arg_p, int *);   /* advance arg_p */
                    Get_buf(fp, buf);
                    sscanf(buf, "%d", ival_p);
                    break;
                case 'f':
                    fval_p = va_arg(arg_p, float *); /* advance arg_p */
                    Get_buf(fp, buf);
                    sscanf(buf, "%f", fval_p);
                    break;
                case 'l':
                    dval_p = va_arg(arg_p, double *);/* advance arg_p */
                    Get_buf(fp, buf);
                    sscanf(buf, "%lf", dval_p);
                    break;
                case 's':
                    sval_p = va_arg(arg_p, char *);  /* advance arg_p */
                    Get_buf(fp, buf);
                    sscanf(buf, "%s", sval_p);
                    break;
                case 'c':
                    cval_p = va_arg(arg_p, char *);  /* advance arg_p */
                    Get_buf(fp, buf);
                    sscanf(buf, "%c", cval_p);
                    break;
            }  /* switch */
        }  /* if */

        fmt_p = Advance(fmt_p);
    }  /* while */

    va_end(arg_p);  /* clean up */
}  /* nscanf */


/*--------------------------------------------------------------*/
/* Use fscanf to read in the next non-comment string, i.e., a
 * sequence of non-whitespace chars whose first character is
 * not COMMENT_CHAR, and the sequence is separated from the rest
 * of the input by whitespace.
 */
void Get_buf(
           FILE*   fp        /* IN/OUT */, 
           char*   buf       /* OUT    */) {
    char *p;

    fscanf(fp, "%s", buf);
#   ifdef DEBUG
    printf("In Get_buf, we read %s\n", buf);
    fflush(stdout);
#   endif
    p = strchr(buf, COMMENT_CHAR);
    while (p == buf) {  /* line with first char = # */
       Go_to_end_of_line(fp);
       fscanf(fp, "%s", buf);
#      ifdef DEBUG
       printf("In Get_buf, we read %s\n", buf);
       fflush(stdout);
#      endif
       p = strchr(buf, COMMENT_CHAR);
    }

    if (p != (char*) NULL) {   /* embedded comment      */
       *p = '\0';
       Go_to_end_of_line(fp);  /* There might be white  */
                               /* space following the # */
    }

}  /*  Get_buf */

/*--------------------------------------------------------------*/
int Whitespace(char c  /* IN */) {
    if ((c == ' ') || (c == '\n') || (c == '\t') )
        return 1;
    else
        return 0;
}  /* Whitespace */


/*--------------------------------------------------------------*/
void Go_to_end_of_line(FILE* fp  /* IN/OUT */) {
   char c;
   fscanf(fp, "%c", &c);
   while (c != '\n') {
      fscanf(fp, "%c", &c);
   }
}  /* Go_to_end_of_line */


/*--------------------------------------------------------------*/
/* Advance buffer pointer to next valid character by
 *     1.  Skipping nonwhite and non-percent (%) characters.
 *     2.  Skipping white space.
 */

char* Advance(char* buf_p /* IN */) {

    char* new_buf_p = buf_p; 

    /* Skip over nonwhite space */
    while ((*new_buf_p != ' ')  && (*new_buf_p != '\t') &&
           (*new_buf_p != '\n') && (*new_buf_p != '%')  &&
           (*new_buf_p != '\0'))
        new_buf_p++;

    /* Skip white space */
    while ((*new_buf_p == ' ')  || (*new_buf_p == '\t') ||
           (*new_buf_p == '\n') )
        new_buf_p++;

    return new_buf_p;
}  /* Advance */
