/* 
 * output.c 
 *
 * Handles output of solution to differential equation.  Writes
 *     data to an output buffer, unless the output buffer would
 *     would overflow.  If the output buffer would overflow,
 *     it writes the data in the buffer to the output file first.
 * In order to allow portability of output files, current version
 *     outputs data in ascii.
 */

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include "mpi.h"
#include "output.h"
#include "equations.h"
#include "utility.h"
#include "globals.h"

/* Maximum size of output buffer in bytes */
/* #define MAX_OUTPUT_BUFSIZ 4096*BUFSIZ  */
/* #define MAX_OUTPUT_BUFSIZ 1024         */
#define MAX_OUTPUT_BUFSIZ 64*BUFSIZ           /* OK for Linux? */

/* Record separator */
#define RECORD_SEP "\n"

/* Struct for storing information about output. */
typedef struct{
    double  *y_final;
    int     index;
    int     record_count;
    int     allocated;
    int     record_length;
    FILE*   outfile;
} de_output_t;

de_output_t output;

#ifdef PROF
extern int print_count;
#endif

#ifdef DIST_OUTPUT
/*===================================================================*/
/* Distributed output */
void Setup_output(void) {

    char   filename[MAX_NAME];
    int    ideal_buf_size;
    double test_buf_size;
    int    print_count;

    /* Allocate a buffer big enough to hold all the output, or, */
    /* MAX_OUTPUT_BUFSIZ bytes.                                 */
    /*     1 variable per neuron                                */
    /*         (membrane voltage)                               */
    /*     * (total_neurons/num_processes)                      */
    /*        = number of doubles per time step                 */
    /*     num_steps + 1 timesteps (+1 for t = 0)               */
    /* First test as a double to avoid arithmetic overflow      */
    print_count = 1 + num_steps/print_freq;
    test_buf_size = (total_neurons/num_processes)*
                        print_count*sizeof(double);

    if ( test_buf_size > ((double) MAX_OUTPUT_BUFSIZ) ) {
        output.y_final = (double*) malloc(MAX_OUTPUT_BUFSIZ);
        output.allocated = MAX_OUTPUT_BUFSIZ/sizeof(double);
    } else { 
        /* Use num_steps+1 to get initial conditions in output */
        ideal_buf_size = (total_neurons/num_processes)*
                           print_count*sizeof(double);
        output.y_final = (double*) malloc(ideal_buf_size);
        output.allocated = ideal_buf_size/sizeof(double);
    }

    if (output.y_final == (double*) NULL) {
        fprintf(stderr, "Malloc of output buffer of size %d failed\n",
                    output.allocated);
        fflush(stderr);
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

    output.index = 0;          /* Where we we are in the buffer   */
                               /* measured in doubles             */

    output.record_count = 0;   /* Where we are in the buffer      */
                               /* measured in records (timesteps) */

    output.record_length = total_neurons/num_processes;

    sprintf(filename, "%s.%d", output_filename, my_rank);
    output.outfile = fopen(filename,"w");
    if (output.outfile == (FILE*) NULL) {
        fprintf(stderr, "Can't open %s for writing\n", filename);
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

}  /* Setup_output */

#else

/*===================================================================*/
/* All output to Process 0 */
void Setup_output(void) {

    int    ideal_buf_size;
    double test_buf_size;
    int    print_count;

    /* Allocate a buffer big enough to hold all the output, or, */
    /* MAX_OUTPUT_BUFSIZ bytes.                                 */
    /*     1 variables per neuron                               */
    /*         (membrane voltage)                               */
    /*     * total_neurons                                      */
    /*     + 1 (for time)                                       */
    /*        = number of doubles per time step                 */
    /*     num_steps + 1 timesteps (+1 for t = 0)               */
    /* First test as a double to avoid arithmetic overflow      */
    print_count = 1 + num_steps/print_freq;
    test_buf_size = (total_neurons + 1.0)*print_count*sizeof(double);

    if ( test_buf_size > ((double) MAX_OUTPUT_BUFSIZ) ) {
        output.y_final = (double*) malloc(MAX_OUTPUT_BUFSIZ);
        output.allocated = MAX_OUTPUT_BUFSIZ/sizeof(double);
    } else { 
        /* Use num_steps+1 to get initial conditions in output */
        ideal_buf_size = (total_neurons + 1)*print_count*sizeof(double);
        output.y_final = (double*) malloc(ideal_buf_size);
        output.allocated = ideal_buf_size/sizeof(double);
    }

    if (output.y_final == (double*) NULL) {
        fprintf(stderr, "Malloc of output buffer of size %d failed\n",
                    output.allocated);
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

    output.index =0;           /* Where we we are in the buffer   */
                               /* measured in doubles             */

    output.record_count = 0;   /* Where we are in the buffer      */
                               /* measured in records (timesteps) */

    output.record_length = total_neurons+1;

    output.outfile = fopen(output_filename,"w");
    if (output.outfile == (FILE*) NULL) {
        fprintf(stderr, "Can't open %s for writing\n", output_filename);
        MPI_Abort(MPI_COMM_WORLD, -1);
    }

}  /* Setup_output */
#endif  /* DIST_OUTPUT */

/*===================================================================*/
void Close_output(void) {

    free(output.y_final);
    fclose(output.outfile);
}  /* Close_output */


#ifdef DIST_OUTPUT
/*===================================================================*/
/* Distributed output -- time is currently unused */
/* Assumes Membrane Voltage is the first (or 0th) equation in each */
/* model.  See below for changes needed if this isn't the case.    */
void Copy_to_output_buffer(
         double        time             /* IN  */,
         double        a[]              /* IN  */) {

    register int     i;
    register double* v_ptr;
    register double* buf_ptr;

    if ( output.index + output.record_length
                      > output.allocated )
        Print_output_buffer();

    buf_ptr = (output.y_final) + (output.index);

    /* If membrane voltage isn't first, add 
     * VOLTAGE_EQN_OFFSETS[neuron_types[my_first_neuron]] to a 
     * to set v_ptr
     */
    v_ptr = a;
    for(i = my_first_neuron; i <= my_last_neuron; i++) {
        *buf_ptr++ = *v_ptr;
        /* If membrane voltage isn't first in each model, subtract
         * VOLTAGE_EQN_OFFSETS[neuron_types[i]] 
         * from v_ptr, add
         * VOLTAGE_EQN_OFFSETS[neuron_types[i+1]] 
         * to v_ptr, and
         * add EQUATION_COUNTS[neuron_types[i]] to 
         * v_ptr.  Note that
         * neuron_types[i+1] may cause a segmentation 
         * violation when
         * i = my_last_neuron
         */
        v_ptr += EQUATION_COUNTS[neuron_types[i]];
    }
 
    (output.index) += (output.record_length);

    (output.record_count)++;

    return;

}  /* Copy_to_output_buffer */

#else

/*===================================================================*/
/* Undistributed output */
/* Assumes Membrane Voltage is the first (or 0th) equation in each */
/* model.  See below for changes needed if this isn't the case.    */
void Copy_to_output_buffer(
         double        time            /* IN  */,
         double        a[]             /* IN  */) {

    register int     i;
    register double* v_ptr;
    register double* buf_ptr;

    if ( output.index + output.record_length
                      > output.allocated )
        Print_output_buffer();

    output.y_final[output.index] = time;

    buf_ptr = (output.y_final) + (output.index) + 1;  
    /* If membrane voltage isn't first, add 
     * VOLTAGE_EQN_OFFSETS[neuron_types[0]] to a to get v_ptr
     */
    v_ptr = a;
    for (i = 0; i < total_neurons;  i++) {
        *buf_ptr++ = *v_ptr;
        /* If membrane voltage isn't first in each model, subtract
         * VOLTAGE_EQN_OFFSETS[neuron_types[i]] from v_ptr, add
         * VOLTAGE_EQN_OFFSETS[neuron_types[i+1]] to v_ptr, and
         * add EQUATION_COUNTS[neuron_types[i]] to v_ptr.  Note that
         * neuron_types[i+1] may cause a segmentation violation when
         * i = total_neurons-1
         */
        v_ptr += EQUATION_COUNTS[neuron_types[i]];
    }
 
    (output.index) += (output.record_length);

    (output.record_count)++;

    return;

}  /* Copy_to_output_buffer */
#endif   /* DIST_OUTPUT */


#ifdef DIST_OUTPUT
/*===================================================================*/
/* Print distributed buffer */
void Print_output_buffer(void) {
    register int  i;
    register int  j;
    register double* buf_ptr;

#ifdef PROF
    print_count++;
#endif

    buf_ptr = output.y_final;
    for (i = 0; i < output.record_count; i++) {

        for (j = 0; j < output.record_length; j++)
            fprintf(output.outfile, "%1.0f ", *buf_ptr++);   
/*          fprintf(output.outfile, "%f ", *buf_ptr++);    */
                    
        fprintf(output.outfile, RECORD_SEP);
    }

    output.record_count = 0;
    output.index = 0;
}  /* Print_output_buffer */

#else 

/*===================================================================*/
/* Print undistributed buffer */
void Print_output_buffer(void) {
    register int  i;
    register int  j;
    register double* buf_ptr;

#ifdef PROF
    print_count++;
#endif

    buf_ptr = output.y_final;
    for (i = 0; i < output.record_count; i++) {

        for (j = 0; j < output.record_length; j++)
            fprintf(output.outfile, "%f ", *buf_ptr++);
                    
        fprintf(output.outfile, RECORD_SEP);
    }

    output.record_count = 0;
    output.index = 0;
}  /* Print_output_buffer */
#endif  /* DIST_OUTPUT */
