/*
 * equations.h
 */

#ifndef EQUATIONS_H
#define EQUATIONS_H

#include "mpi.h"
#include "comm.h"

/*
 * Parameters needed to specify the equations modelling the neuron.
 * These are defined in equations.c.
 */

/* This macro defines how synaptic currents are exchanged among
 * neurons belonging to different processes.  Current (7/29/98)
 * possible values:
 *
 *     1.  MPI_Allgather:  use this if there are synaptic connections
 *         between arbitrary neurons.
 *     2.  Pairwise_exchange:  use this if there are only synaptic
 *         connections between neurons assigned to adjacent processes
 *         and buffers are contiguous.
 */
/* #define ALL_GATHER Pairwise_exchange */
#define ALL_GATHER MPI_Allgather

/* Number of input int constants */
#define INT_COUNT 0
/* Number of input double constants */
#define DOUBLE_COUNT 19

/* List of input constants from equations.c */
extern double 
    SIGMA,
    G_NA,
    G_K,
    G_L,
    V_NA,
    V_K,
    V_L,
    V_IE,
    V_EE,
    V_II,
    V_EI,
    G_EI,
    G_IE,
    G_II,
    AE0,
    AEF,
    AI,
    SIGMOID_THETA,
    SIGMOID_POWER;

/* Stores information about synaptic connections */
/* Shouldn't need to be changed                  */
typedef struct { 
    int dimension; 
    int* links; 
    int* signs; /* added for connection types */
} syn_conn_t;


/* Function Pointer type.  Probably don't need this if you only */
/* have one model                                               */
typedef void (*model_fp) (
             double      y[]                /* IN  */,
             double      answer[]           /* OUT */,
             double      syn_input[]        /* IN  */,
             double      syn_answer[]       /* OUT */,
             int         i                  /* IN  */,
             int         neuron             /* IN  */,
             int         global_neuron_rank /* IN  */);

/* Make this data available to other parts of the program */
/* Shouldn't need modification                            */
extern int      MODEL_COUNT;
extern int      EQUATION_COUNTS[];
extern int      VOLTAGE_EQN_OFFSETS[];
extern int      SYNAPTIC_EQN_OFFSETS[];

/* Modify these prototypes to agree with your model.  If you */
/* change the argument lists for Setup_f, Compute_f, or      */
/* Cleanup_f, you'll need to change the calls in the solver  */
void Setup_f(void);

void Compute_f(
         double       y[]               /* IN  */,
         double       syn_input[]       /* IN  */,
         double       answer[]          /* OUT */,
         double       syn_answer[]      /* OUT */);

void Model_E (
         double      y[]                /* IN  */,
         double      answer[]           /* OUT */,
         double      syn_input[]        /* IN  */,
         double      syn_answer[]       /* OUT */,
         int         i                  /* IN  */,
         int         neuron             /* IN  */,
         int         global_neuron_rank /* IN  */);

void Compute_V_prime_E(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         double       syn_input[]        /* IN  */,
         int          i                  /* IN  */,
         int          neuron             /* IN  */);

void Compute_h_prime_E(
         double   y[]       /* IN  */,
         double   answer[]  /* OUT */,
         int      i         /* IN  */);

void Compute_n_prime_E(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         int          i                  /* IN  */);

void Compute_m_prime_E(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         int          i                  /* IN  */);

void Compute_s_prime_E(
         double   y[]                 /* IN  */,
         double   syn_input[]         /* IN  */,
         double   syn_answer[]        /* OUT */,
         int      i                   /* IN  */,
         int      neuron              /* IN  */,
         int      global_neuron_rank  /* IN  */);

void Model_I (
         double      y[]                /* IN  */,
         double      answer[]           /* OUT */,
         double      syn_input[]        /* IN  */,
         double      syn_answer[]       /* OUT */,
         int         i                  /* IN  */,
         int         neuron             /* IN  */,
         int         global_neuron_rank /* IN  */);

void Compute_V_prime_I(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         double       syn_input[]        /* IN  */,
         int          i                  /* IN  */,
         int          neuron             /* IN  */);

void Compute_h_prime_I(
         double   y[]       /* IN  */,
         double   answer[]  /* OUT */,
         int      i         /* IN  */);

void Compute_n_prime_I(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         int          i                  /* IN  */);

void Compute_m_prime_I(
         double       y[]                /* IN  */,
         double       answer[]           /* OUT */,
         int          i                  /* IN  */);

void Compute_s_prime_I(
         double   y[]                 /* IN  */,
         double   syn_input[]         /* IN  */,
         double   syn_answer[]        /* OUT */,
         int      i                   /* IN  */,
         int      neuron              /* IN  */,
         int      global_neuron_rank  /* IN  */);

void Cleanup_f(void);
#endif
