/* 
 * File:     inputgen.c 
 *
 * Purpose:  Generate input file for solver.  The generated file will 
 *    contain the following information:
 *        initial time value
 *        step size
 *        number of steps
 *        number of variables in system modelling neuronal network
 *        initial conditions for variables
 *        frequency with which solver prints results
 *        name of output file to which solver prints 
 *        information on neuronal network
 *    The neurons in the generated neuronal network form a rectangular
 *    grid with randomly generated synaptic connections, randomly
 *    generated model types, and randomly generated "signs" on
 *    interconnections -- a connection can be either excitatory (+)
 *    or inhibitory (-).  The type of a neuron specifies which
 *    model will be used for the neuron.  If, for example, there
 *    are three model types, then a neurons will have type 0, 1, or 2.
 *
 *    The neurons are identified with vertices in a rectangular
 *    grid in which adjacent vertices in the same row or same column
 *    are 1 unit apart.  So the "distance" between two neurons is just 
 *    the euclidean distance between the corresponding vertices.  In 
 *    order to determine whether two neurons are interconnected, a
 *    random value in the range [0,1) is generated.  If this value
 *    is less than a decaying exponential of the square of the distance 
 *    between them, exp[-alpha*distance*distance], then a connection
 *    is generated.  The program attempts to generate connections
 *    between all pairs of neurons.  So there may be a connection
 *    from neuron A to neuron B, but not from neuron B to neuron A.  
 *
 * Input:    
 *     'd' or 's':       input in datafile or typed from standard input
 *     if 'd':           filename
 *     t_0:              initial time value (double)
 *     h:                stepsize (double)
 *     N:                number of steps (int)
 *     row_count:        number of rows in grid of neurons (int)
 *     col_count:        number of columns in grid of neurons (int)
 *     print_freq:       the frequency that the solver results; e.g.,
 *                       print_freq = 10, solver prints results for
 *                       time steps 0, 10, 20, etc.
 *     outfile:          name of file to which solver should print
 *                       results.
 *     alpha:            value controlling the density of neural 
 *                       interconnections:  larger values of
 *                       alpha, reduce the number of interconnections
 *                       (double).  
 *     prob:             the probability that a synaptic connection
 *                       is excitatory.  (double)
 *     max_procs:        the maximum number of processes on which the
 *                       the solver will be run:  used to guarantee that
 *                       neural interconnections won't be more than
 *                       one process distant.  (Can ignore this by
 *                       setting max_procs = 1.)
 *     model_count:      the number of neuron models
 *     var_counts:       the number of variables for each model,
 *                       an array of model_count ints.  N.B., these
 *                       counts should *include* synaptic currents.
 *     model_probs:      relative probabilities of the neuron models.
 *                       an array of model_count ints.  For example,
 *                       if there are 3 models and the values are
 *                       2, 1, 1, then model 0 should occur twice
 *                       as often as models 1 and 2.
 *                       
 * 
 * Output:  solver_infile. File containing input data for solver. It
 *     contains:
 *     t_0, h, N, total number of vars (including synaptic currents),
 *     neuron_count, row_count, col_count,
 *     print_freq, outfile,
 *     neuron_types array
 *     initial conditions (see note 3), 
 *     randomly generated adjacency list:  This consists of two
 *         parts: 
 *             count of the number of synaptic inputs to each neuron
 *             for each neuron, a list of pairs.  Each pair corresponds
 *                 to a synaptic input to the neuron.  The first number
 *                 in the pair is the rank of the neuron from which
 *                 the synaptic input is received.  The second number 
 *                 is the sign of the synaptic input:  +1 for excitatory
 *                 or -1 for inhibitory.
 *
 * Notes:
 *     1.  There is no sanity check on the input data.
 *     2.  This version generates a random value in the range
 *         -100 to +50 for the first variable (membrane voltage)
 *         associated with each neuron and 0's for all other initial 
 *         values.  Modify Gen_initial_values and Initial_value to change
 *         this.
 */ 

#include <stdio.h> 
#include <stdlib.h> 
#include <math.h> 
#include <time.h> 

/* -------------------------------------------------------- */
#define MAX_NAME 256

double Calc_distance(int sourceId   /* IN */, 
                     int destId     /* IN */,
                     int col_count  /* IN */);

void Skip_white_space(void);

void Read_data(double*  time_init    /* OUT */,
               double*  step_size    /* OUT */,
               int*     num_steps    /* OUT */,
               int*     row_count    /* OUT */,
               int*     col_count    /* OUT */,
               int*     print_freq   /* OUT */,
               char*    outfile      /* OUT */,
               double*  alpha        /* OUT */,
               double*  prob         /* OUT */,
               int*     max_procs    /* OUT */,
               int*     model_count  /* OUT */, 
               int**    var_counts   /* OUT */, 
               int**    model_probs  /* OUT */);

void Gen_model_types(int total_neurons   /* IN  */, 
                     int model_count     /* IN  */, 
                     int model_probs[]   /* IN  */,
                     int neuron_types[]  /* OUT */);

int Get_eqn_count(int total_neurons   /* IN  */, 
                  int neuron_types[]  /* IN  */,
                  int var_counts[]    /* IN */);

int Flip_coin(double prob  /* IN */);

int Is_connected(double alpha     /* IN */, 
                 double distance  /* IN */);

void Gen_list(int*   dimensions   /* OUT */, 
              int**  link_type    /* OUT */, 
              int    row_count    /* IN  */, 
              int    col_count    /* IN  */, 
              double alpha        /* IN  */, 
              double prob         /* IN  */,
              int    max_procs    /* IN  */);
 
void Print_double_vector(FILE*     fp                 /* IN */,
                         double*   vect               /* IN */,
                         int       size               /* IN */);

void Print_int_vector( FILE*     fp                 /* IN */,
                       int*      vect               /* IN */,
                       int       size               /* IN */);

void Gen_initial_values(int     total_neurons    /* IN  */, 
                        int     neuron_types[]   /* IN  */,
                        int     var_counts[]     /* IN  */,
                        double  initial_values[] /* OUT */);

double Initial_value(int model_type  /* IN */,
                     int variable    /* IN */);

void Compute_dest_limits(
         int   neuron             /* IN  */, 
         int   row_count    /* IN  */, 
         int   col_count    /* IN  */,
         int   max_procs          /* IN  */, 
         int*  min_dest_neuron    /* OUT */, 
         int*  max_dest_neuron    /* OUT */);


/* -------------------------------------------------------- */
int main() { 
    double  time_init;
    double  step_size;
    int     num_steps;
    int     row_count;
    int     col_count;
    int     print_freq;
    char    outfile[MAX_NAME];
    double  alpha;
    double  prob;
    int     max_procs;
    int     model_count;
    int*    var_counts;
    int*    model_probs;
 
    int     total_neurons;
    int     total_equations;
    int*    neuron_types;
    int*    dimensions;  /* Number of synaptic inputs to each neuron */
    int**   link_type;
    int     i;
 
    double* initial_values;
  
    FILE*   fp;
 
    srand48(0);
 
    /* read in data */
    Read_data(&time_init, &step_size, &num_steps, &row_count,
              &col_count, &print_freq, outfile, &alpha, &prob,
              &max_procs, &model_count, &var_counts, &model_probs);
 
    total_neurons = row_count*col_count;
    neuron_types = (int*) malloc(total_neurons*sizeof(int));
    Gen_model_types(total_neurons, model_count, model_probs,
        neuron_types);
    total_equations = Get_eqn_count(total_neurons, neuron_types,
        var_counts);
 
    dimensions = (int*) malloc(sizeof(int)*total_neurons); 
    for (i = 0; i < total_neurons; i++)
        dimensions[i] = 0;
 
    link_type = (int**) malloc(sizeof(int*)*total_neurons);
 
    Gen_list(dimensions, link_type, row_count, col_count,
            alpha, prob, max_procs);
 
    initial_values = 
        (double*) malloc(sizeof(double)*total_equations);
 
    Gen_initial_values(total_neurons, neuron_types, var_counts,
        initial_values);
 
    /* Output to file for solver */
    fp = fopen("solver_infile", "w");
    if (!fp) {
        fprintf(stderr, "Attempt to open solver_infile failed.\n");
        free(dimensions);
        for (i = 0; i < total_neurons; i++)
            free(link_type[i]);
        free(link_type);
        free(initial_values);
        free(var_counts);
        free(model_probs);
        free(neuron_types);
        exit(-1);
    }
 
    fprintf(fp, "%f\n", time_init);
    fprintf(fp, "%f\n", step_size);
    fprintf(fp, "%d\n", num_steps);
    fprintf(fp, "%d\n", total_equations);
    fprintf(fp, "%d\n", total_neurons);
    fprintf(fp, "%d\n", row_count);
    fprintf(fp, "%d\n", col_count);
    fprintf(fp, "%d\n", print_freq);
    fprintf(fp, "%s\n", outfile);
    Print_int_vector(fp, neuron_types, total_neurons);
    Print_double_vector(fp, initial_values, 
                        total_equations);
    Print_int_vector(fp, dimensions, total_neurons);
    for (i = 0; i < total_neurons; i++) {
        Print_int_vector(fp, link_type[i], (dimensions[i]) * 2);
    }
    
    fclose(fp);
 
    free(dimensions);
    for (i = 0; i < total_neurons; i++)
        free(link_type[i]);
    free(link_type);
    free(initial_values);
    free(var_counts);
    free(model_probs);
    free(neuron_types);
 
    return 0;
    
}  /* main */


/*-----------------------------------------------------------------*/
void Read_data(double*  time_init    /* OUT */,
               double*  step_size    /* OUT */,
               int*     num_steps    /* OUT */,
               int*     row_count    /* OUT */,
               int*     col_count    /* OUT */,
               int*     print_freq   /* OUT */,
               char*    outfile      /* OUT */,
               double*  alpha        /* OUT */,
               double*  prob         /* OUT */,
               int*     max_procs    /* OUT */,
               int*     model_count  /* OUT */, 
               int**    var_counts   /* OUT */, 
               int**    model_probs  /* OUT */) {
    char which_one;
    char filename[MAX_NAME];
    FILE* fp;
    int i;
 
    printf("Dear user:  Data file or standard in? ['d' or 's']\n");
    Skip_white_space();
    scanf("%c", &which_one);
    if (which_one == 's') {
        fp = stdin;
    } else {
        printf("What's the file name?\n");
        scanf("%s", filename);
        fp = fopen(filename,"r");
    }
 
    if (!fp) {
        fprintf(stderr,"Attempt to open %s failed.\n", filename);
        exit(-1);
    }
 
    if (which_one == 's') {
        printf("Please enter t_0, h, N, row_count, ");
        printf("col_count, print_freq, outfile, alpha\n");
        printf("prob, max_procs, ");
        printf("model_count, var_counts, and model_probs\n");
    }
    fscanf(fp, "%lf", time_init);
    printf("t_0 = %f\n", *time_init);
    fscanf(fp, "%lf", step_size);
    printf("h = %f\n", *step_size);
    fscanf(fp, "%d", num_steps);
    printf("num_steps = %d\n", *num_steps);
    fscanf(fp, "%d", row_count);
    printf("row_count = %d\n", *row_count);
    fscanf(fp, "%d", col_count);
    printf("col_count = %d\n", *col_count);
    fscanf(fp, "%d", print_freq);
    printf("print_freq = %d\n", *print_freq);
    fscanf(fp, "%s", outfile);
    printf("outfile = %s\n", outfile);
    
    fscanf(fp, "%lf", alpha);
    printf("alpha = %f\n", *alpha);
    fscanf(fp, "%lf", prob);
    printf("prob = %f\n", *prob);
    fscanf(fp, "%d", max_procs);
    printf("max_procs = %d\n", *max_procs);
 
    fscanf(fp, "%d", model_count);
    printf("model_count = %d\n", *model_count);
    *var_counts = (int *) malloc((*model_count)*sizeof(int));
    if (*var_counts == (int *) NULL) {
        fprintf(stderr, "Can't allocate var_counts\n");
        exit(-1);
    }
    *model_probs = (int *) malloc((*model_count)*sizeof(int));
    if (*model_probs == (int *) NULL) {
        fprintf(stderr, "Can't allocate model_probs\n");
        exit(-1);
    }
    for (i = 0; i < *model_count; i++)
        fscanf(fp, "%d", &((*var_counts)[i]));
    printf("var_counts = ");
    Print_int_vector(stdout, *var_counts, *model_count);
    for (i = 0; i < *model_count; i++)
        fscanf(fp, "%d", &((*model_probs)[i]));
    printf("model_probs = ");
    Print_int_vector(stdout, *model_probs, *model_count);

    fclose(fp);
}  /* Read_data */


/*----------------------------------------------------------*/
void Skip_white_space(void) {
    char c;

    c = getchar();
    while ((c == ' ') || (c == '\n') || (c == '\t'))
        c = getchar();
    ungetc(c, stdin);
}  /* Skip_white_space */


/*----------------------------------------------------------*/
void Gen_model_types(int total_neurons   /* IN  */, 
                     int model_count     /* IN  */, 
                     int model_probs[]   /* IN  */,
                     int neuron_types[]  /* OUT */) {

    int* cumulative_probs;
    int modulus;
    int rand_val;
    int i, j;

    cumulative_probs = (int*) malloc(model_count*sizeof(int));
    cumulative_probs[0] = model_probs[0];
    for (i = 1; i < model_count; i++)
        cumulative_probs[i] = cumulative_probs[i-1] + model_probs[i];
    modulus = cumulative_probs[model_count-1];

    for (i = 0; i < total_neurons; i++) {
        rand_val = lrand48() % modulus;
        j = 0;
        while (rand_val >= cumulative_probs[j]) j++;
        neuron_types[i] = j;
    }

    free(cumulative_probs);
}  /* Gen_model_types */


/* -------------------------------------------------------- */
int Get_eqn_count(int total_neurons   /* IN  */, 
                  int neuron_types[]  /* IN  */,
                  int var_counts[]    /* IN */) {

    int total = 0;
    int i;

    for (i = 0; i < total_neurons; i++)
        total += var_counts[neuron_types[i]];

    return total;

}  /* Get_eqn_count */


/* -------------------------------------------------------- */
void Gen_list(int*   dimensions   /* OUT */, 
              int**  link_type    /* OUT */, 
              int    row_count    /* IN  */, 
              int    col_count    /* IN  */, 
              double alpha        /* IN  */, 
              double prob         /* IN  */,
              int    max_procs    /* IN  */) {
 
    int i, j;
    double dist;
    int total_neurons;

    int min_dest_neuron;
    int max_dest_neuron;

    int* temp_link_type;
  
    total_neurons = row_count * col_count;

    temp_link_type = (int*) malloc(2*total_neurons*sizeof(int));
    if (temp_link_type == (int*) NULL) {
        fprintf(stderr, "Can't allocate temp_link_type\n");
        exit(-1);
    }

    for(i = 0; i < total_neurons; i++) {
        Compute_dest_limits(i, row_count, col_count,
            max_procs, &min_dest_neuron, &max_dest_neuron);

        for(j = min_dest_neuron; j <= max_dest_neuron; j++) {
            if (j != i) {
                dist = Calc_distance(i, j, col_count);
                if (Is_connected(alpha, dist) == 1) {
/*                  (link_type[i])[(dimensions[i]) * 2] = j; */
                    temp_link_type[2*dimensions[i]] = j;
                    if (Flip_coin(prob) == 1) {
/*                      (link_type[i])[(dimensions[i]) *2 +1] = 1; */
                        temp_link_type[2*dimensions[i] + 1] = 1;
                    } else {
/*                      (link_type[i])[(dimensions[i]) *2 +1] = -1; */
                        temp_link_type[2*dimensions[i] + 1] = -1;
                    }
                    dimensions[i]++;
                }  /* if Is_connected */
            }
        }  /* for j */
        if (dimensions[i] > 0) {
            link_type[i] = (int*) malloc(2*dimensions[i]*sizeof(int));
            if (link_type[i] == (int*) NULL) {
                fprintf(stderr, "Can't allocate link_type[%d]\n", i);
                exit(-1);
            }
            memcpy(link_type[i], temp_link_type, 
                       2*dimensions[i]*sizeof(int));
        }
    }  /* for i */


    free(temp_link_type);
}  /* Gen_list */


/*---------------------------------------------------------*/
int Is_connected(double alpha     /* IN */, 
                 double distance  /* IN */) {

   double cutoff, rand_val; 

   cutoff = exp((-1)*alpha*distance*distance); 
   rand_val = drand48();
   if (rand_val < cutoff) 
       return 1; 
   else 
       return 0; 
}  /* Is_connected */


/*---------------------------------------------------------*/ 
int Flip_coin(double prob  /* IN */) { 
   double rand_val; 

   rand_val = drand48();
   if (rand_val < prob) 
      return 1; 
   else 
      return 0; 
}  /* Flip_coin */ 

 
/*---------------------------------------------------------*/
double Calc_distance(int sourceId   /* IN */, 
                     int destId     /* IN */,
                     int col_count  /* IN */) { 
   double distX, distY, distance; 
   double sourceX, sourceY, destX, destY; 

   /* X = column number, Y = row number */

   sourceX = (double) (sourceId % col_count); 
   sourceY = (double) (sourceId/col_count); 
   destX = (double) (destId % col_count); 
   destY = (double) (destId/col_count); 
   distX = destX - sourceX; 
   distY = destY - sourceY; 
   distance = distX*distX + distY*distY; 
   distance = sqrt( distance ); 
   return distance; 
}   /* Calc_distance */


/* -------------------------------------------------------- */ 
void Print_double_vector(
         FILE*     fp                 /* IN */,
         double*   vect               /* IN */,
         int       size               /* IN */) {
    int i;

    for (i = 0; i < size; i++)
          fprintf(fp, "%f ", vect[i]);
    fprintf(fp, "\n");
}  /* Print_double_vector */


/*---------------------------------------------------------*/
void Print_int_vector(
         FILE*     fp                 /* IN */,
         int*      vect               /* IN */,
         int       size               /* IN */) {
    int i;

    for (i = 0; i < size; i++)
          fprintf(fp, "%d ", vect[i]);
    fprintf(fp, "\n");
}  /* Print_int_vector */


/*-------------------------------------------------------*/
void Gen_initial_values(int     total_neurons    /* IN  */, 
                        int     neuron_types[]   /* IN  */,
                        int     var_counts[]     /* IN  */,
                        double  initial_values[] /* OUT */) {

    int i, j, index = 0;

    for(i = 0; i < total_neurons; i++) {
        for (j = 0; j < var_counts[neuron_types[i]]; j++) {
            initial_values[index] = Initial_value(neuron_types[i], j);
            index++;
        }
    } /* for i */

}  /* Gen_initial_values */


/*-----------------------------------------------------*/
/* Assign 0 to all variables. */
double Initial_value(int model_type  /* IN */,
                     int variable    /* IN */) {

    switch (model_type) {
        default:
            switch(variable) {
                case 0:
                    return (150.0*drand48() - 100.0);
                    break;
                default:
                    return 0.0;
            }
    }

}  /* Initial_value */


/*-----------------------------------------------------*/
void Compute_dest_limits(
            int   neuron             /* IN  */, 
            int   row_count          /* IN  */, 
            int   col_count          /* IN  */,
            int   max_procs          /* IN  */, 
            int*  min_dest_neuron    /* OUT */, 
            int*  max_dest_neuron    /* OUT */) {

    static int first_time = 1;
    static int rows_per_proc;

    int my_row;
    int my_proc;
    int max_row_on_my_proc;
    int min_row_on_my_proc;
    int min_dest_row;
    int max_dest_row;

    if (first_time) {
        rows_per_proc = row_count/max_procs;
        first_time = 0;
    }

    my_row = neuron/row_count;
    my_proc = my_row/rows_per_proc;
    min_row_on_my_proc = my_proc*rows_per_proc;
    max_row_on_my_proc = min_row_on_my_proc + rows_per_proc - 1;
    min_dest_row = min_row_on_my_proc - rows_per_proc;
    if (min_dest_row < 0) min_dest_row = 0;
    max_dest_row = max_row_on_my_proc + rows_per_proc;
    if (max_dest_row >= col_count) 
        max_dest_row = col_count - 1;

    *min_dest_neuron = min_dest_row*row_count;
    *max_dest_neuron = 
        max_dest_row*row_count + row_count - 1;

}  /* Compute_dest_limits */
