/* interpreter.c */ 

#include <stdio.h> 
#include <stdlib.h> 
#include <string.h>
#include <ctype.h>
#include "split_defs.h" 
#include "split_util.h" 
#include "interpreter.h" 

/*-------------------------------------------------------------------*/ 
void ReadInput( int argc, char** argv, struct parameters* p ) 
{ 
   char** com_strings; 
   int vars_per_neuron; 
   long int num_neurons; 

   GetComStrings( argc, argv, &com_strings ); 
   CheckComInfo( com_strings ); 

   /* get int variables */ 
   p->num_pieces   = atoi( com_strings[ NP_IND ] ); 
   vars_per_neuron = atoi( com_strings[ VN_IND ] ); 
   p->num_lines    = atoi( com_strings[ NL_IND ] ); 
   num_neurons = ConvertExpr( com_strings[ NN_IND ], vars_per_neuron ); 

   /* check validity of int values */ 
   CheckDivisibility( (p->num_pieces), vars_per_neuron, num_neurons ); 

   /* calculate one of the parameters */ 
   p->vars_per_piece = num_neurons*(long)vars_per_neuron; 
   p->vars_per_piece/=(p->num_pieces);

   /* get string variables */ 
   strcpy( p->ifname, com_strings[ IF_IND ] ); 
   strcpy( p->ofname, com_strings[ OF_IND ] ); 
   strcat( p->ofname, "." ); 
   if( strcmp( com_strings[ FF_IND ], "bin" ) == 0 ) 
      p->ftype = BINARY; 
   else 
      p->ftype = TEXT; 

   ClearStrings( com_strings ); 
}

/*-------------------------------------------------------------------*/ 
void GetComStrings( int argc, char** argv, char*** coms ) 
{ 
   switch( argc ) 
   { 
      case 1: (*coms) = HandleArgc1();         return; 
      case 2: HandleArgc2( argv[ 1 ] );        return; 
      case 3: (*coms) = HandleArgc3( argv );   return; 
      case 8: (*coms) = argv;                  return; 

      default: break; 
   } 

   fprintf( stderr, "\nCannot execute command!\n" ); 
   fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" ); 
   exit( 1 ); 
} 

/*-------------------------------------------------------------------*/ 
char** HandleArgc1() 
{ 
   FILE* fp; 
   char** coms; 

   fp = fopen( SPLITFILE, READ ); 
   if( fp == NULL ) 
      coms = GetComsFromUser(); 
   else 
   { 
      coms = GetComsFromFile( fp ); 
      fclose( fp ); 

      if( coms == NULL )
      {
         fprintf( stderr, "\nInsufficient info in \"%s\"!\n", SPLITFILE ); 
         fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
         exit( 1 );
      }
   } 

   return coms; 
} 

/*-------------------------------------------------------------------*/ 
void HandleArgc2( char* opt ) 
{ 
   int i; 

   for( i = 0; opt[ i ] != '\0'; i++ ) 
      opt[ i ] = tolower( opt[ i ] ); 

   if( strcmp( opt, USAGE ) == 0 ) 
   { 
      Usage(); 
      exit( 1 ); 
   } 

   if( strcmp( opt, READOPT ) == 0 )
      fprintf( stderr, "\nFilename missing.\n" );
   else if( strcmp( opt, USAGE ) != 0 )
      fprintf( stderr, "\nUnrecognized option : %s\n", opt );
   fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
   exit( 1 );
} 

/*-------------------------------------------------------------------*/ 
char** HandleArgc3( char** argv ) 
{ 
   FILE* fp; 
   char** coms; 

   if( strcmp( argv[ 1 ], READOPT ) != 0 )
   {
      fprintf( stderr, "\nUnrecognized option : %s\n", argv[ 1 ] );
      fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
      exit( 1 );
   }

   fp = OpenFile( argv[ 2 ], READ ); 
   coms = GetComsFromFile( fp ); 
   fclose( fp ); 

   if( coms == NULL ) 
   {
      fprintf( stderr, "\nInsufficient info in \"%s\"!\n", argv[ 2 ] );
      fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
      exit( 1 );
   }

   return coms; 
} 

/*-------------------------------------------------------------------*/ 
char** GetComsFromFile( FILE* fp ) 
{ 
   char** tmp; 
   char* str; 
   int i; 

   /* initialize string array */
   tmp = (char**)malloc( MAXARGC*sizeof( char* ) );
   for( i = 0; i < MAXARGC; i++ )
      tmp[ i ] = NULL;

   /* read info in resource file */
   str = GetString( fp );
   for( i = 1; str != NULL && i < MAXARGC; i++ )
   {
      tmp[ i ] = str;
      str = GetString( fp );
   }

   if( i < MAXARGC )
   {
      ClearStrings( tmp ); 
      return NULL; 
   } 

   return tmp; 
} 

/*-------------------------------------------------------------------*/
char** GetComsFromUser() 
{ 
   char** tmp; 
   char str[ MAX ]; 
   int i; 

   /* initialize string array */
   tmp = (char**)malloc( MAXARGC*sizeof( char* ) );
   for( i = 0; i < MAXARGC; i++ )
      tmp[ i ] = (char*)malloc( MAX*sizeof( char ) );

   /* prompt user */ 
   printf( "\ninput file name : " ); 
   scanf( "%s", str ); 
   strcpy( tmp[ IF_IND ], str ); 
   printf( "output file name : " );
   scanf( "%s", str );
   strcpy( tmp[ OF_IND ], str );
   printf( "file format type : " );
   scanf( "%s", str );
   strcpy( tmp[ FF_IND ], str );
   printf( "number of pieces : " );
   scanf( "%s", str );
   strcpy( tmp[ NP_IND ], str );
   printf( "number of neurons or of values per line : " );
   scanf( "%s", str );
   strcpy( tmp[ NN_IND ], str );
   printf( "number of variables per neuron : " );
   scanf( "%s", str );
   strcpy( tmp[ VN_IND ], str );
   printf( "number of lines : " );
   scanf( "%s", str );
   strcpy( tmp[ NL_IND ], str );

   return tmp; 
} 

/*-------------------------------------------------------------------*/ 
void CheckComInfo( char** com_strings ) 
{ 
   int i; 

   for( i = 4; i < MAXARGC; i++ ) 
   { 
      if( i == NN_IND ) 
         CheckComplexStr( com_strings[ i ] ); 
      else 
      { 
         if( !isint( com_strings[ i ] ) ) 
         { 
            fprintf( stderr, "\nInvalid input : %s\n", com_strings[ i ] );
            fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
            exit( 1 );
         } 
      } 
   } 
} 

/*-------------------------------------------------------------------*/ 
void CheckComplexStr( char* expr ) 
{ 
   char tmp[ MAX ]; 
   int i; 

   /* take out the first char of string expr */ 
   for( i = 1; expr[ i ] != EOS; i++ ) 
      tmp[ i - 1 ] = expr[ i ]; 
   tmp[ i - 1 ] = EOS; 
   
   /* test if values_per_line is valid */ 
   if( expr[ 0 ] == 'v' ) 
      if( isint( tmp ) ) 
         return; 

   /* test if num_neurons is valid */ 
   if( expr[ 0 ] == 'n' ) 
   { 
      if( expr[ 1 ] != LPARENTH ) 
      { 
         if( isint( tmp ) ) 
            return; 
      } 
      else 
      { 
         if( IsValidExpr( tmp ) ) 
            return; 
      } 
   } 

   fprintf( stderr, "\nInvalid input : %s\n", expr );
   fprintf( stderr, "Type \"splitter -u\" for usage.\n\n" );
   exit( 1 );
} 

/*-------------------------------------------------------------------*/ 
int IsValidExpr( char* expr ) 
{ 
   char row[ MAX ];
   char col[ MAX ];
   char op[ MAX ];
   int i = 1;
   int j = 0;

   for( j = 0; isdigit( expr[ i ] ) && expr[ i ] != EOS; i++, j++ )
      row[ j ] = expr[ i ];
   row[ j ] = EOS;

   for( j = 0; !isdigit( expr[ i ] ) && expr[ i ] != EOS; i++, j++ )
      op[ j ] = expr[ i ];
   op[ j ] = EOS;

   for( j = 0; expr[ i ] != RPARENTH && expr[ i ] != EOS; i++, j++ )
      col[ j ] = expr[ i ];
   col[ j ] = EOS;

   if( isint( row ) && isint( col ) ) 
   { 
      if( strcmp( op, BY ) == 0   || 
          strcmp( op, MULT ) == 0 || 
          strcmp( op, TIMES ) == 0 ) 
      { 
         if( expr[ i ] == RPARENTH && expr[ i + 1 ] == EOS ) 
            return 1; 
      } 
   } 

   return 0; 
} 

/*-------------------------------------------------------------------*/ 
long ConvertExpr( char* expr, int vars_per_neuron ) 
{ 
   char valA[ MAX ]; 
   char valB[ MAX ]; 
   long int r, c, n; 
   int i, j; 

   if( isdigit( expr[ 1 ] ) ) 
   { 
      /* copy the string to convert to long */ 
      for( i = 1, j = 0; expr[ i ] != EOS; i++, j++ ) 
         valA[ j ] = expr[ i ]; 
      valA[ j ] = EOS; 
      n = atol( valA ); 

      /* return number of neurons */ 
      if( expr[ 0 ] == 'v' ) 
         return ( n/(long int)vars_per_neuron ); 
      else   /* expr[ 0 ] == 'n' */ 
         return n; 
   } 

   /* copy string indicating number of rows */ 
   for( i = 2, j = 0; isdigit( expr[ i ] ); i++, j++ ) 
      valA[ j ] = expr[ i ]; 
   valA[ j ] = '\0'; 

   /* skip to next digit string */ 
   while( !isdigit( expr[ i ] ) ) 
      i++; 

   /* copy string indicating number of columns */ 
   for( j = 0; expr[ i ] != RPARENTH; i++, j++ ) 
      valB[ j ] = expr[ i ]; 
   valB[ j ] = '\0';  

   /* convert strings to long         */ 
   /* and calculate number of neurons */ 
   r = atol( valA ); 
   c = atol( valB ); 
   n = r*c; 

   return n; 
} 

/*-------------------------------------------------------------------*/ 
void CheckDivisibility( int pc, int vn, long nn ) 
{ 
   long vl;   /* values_per_line */ 
   int mod; 
   char sel; 

   /* values_per_line = vars_per_neuron * num_neurons */ 
   vl = (long)vn*nn; 
   mod = (int)( vl%(long)pc ); 

   if( mod != 0 ) 
   { 
      fprintf( stderr, "\nError : File cannot be " ); 
      fprintf( stderr, "divided into %d pieces.\n\n", pc ); 
      exit( 1 ); 
   } 

   if( mod == 0 && nn%(long)pc != 0 ) 
   { 
      printf( "\nWarning : This file division " ); 
      printf( "ignores neuron partition.\n" ); 
      printf( "Would you like to proceed? (y/n): " ); 
      scanf( "%c", &sel ); 
      while( sel != 'Y' && sel != 'y' && sel !='N' && sel != 'n' ) 
      { 
         printf( "Invalid input. Try again. (y/n): " ); 
         scanf( "%c", &sel ); 
         scanf( "%c", &sel ); 
      } 
      if( sel == 'N' || sel == 'n' ) 
      { 
         printf( "\nExecution terminated!\n\n" ); 
         exit( 1 ); 
      } 
   } 
} 

/*-------------------------------------------------------------------*/ 
void ClearStrings( char** strs ) 
{ 
   int i; 

   for( i = 0; i < MAXARGC; i++ ) 
      free( strs[ i ] ); 
   free( strs ); 
} 

/*-------------------------------------------------------------------*/ 

