/*
* Copyright (c) 2006, 2007 Sun Microsystems, Inc.
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to 
* deal in the Software without restriction, including without limitation the 
* rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
* sell copies of the Software, and to permit persons to whom the Software is 
* furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in 
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
 **/       
package org.sunspotworld.demo;

/*
 * AccelerometerSampleCode.java
 *
 * Simple use of the accelerometer to measure tilt on the Sun SPOT.
 *
 * author: Ron Goldman  
 * date: August 14, 2006
 * date: March 12, 2007 - modified to use new IAccelerometer3D interface
 */

import com.sun.spot.sensorboard.EDemoBoard;
import com.sun.spot.sensorboard.peripheral.ITriColorLED;
import com.sun.spot.sensorboard.peripheral.LEDColor;
import com.sun.spot.sensorboard.peripheral.IAccelerometer3D;
import com.sun.spot.util.Utils;
import com.sun.squawk.util.MathUtils;

import java.io.IOException;

import javax.microedition.midlet.MIDlet;
import javax.microedition.midlet.MIDletStateChangeException;

/**
 * Sample code snippet to show how to use the accelerometer
 * on the Sun SPOT General Purpose Sensor Board.
 *
 * @author Ron Goldman
 */
public class AccelerometerSampleCode extends MIDlet {

    private IAccelerometer3D accel = EDemoBoard.getInstance().getAccelerometer();
    private ITriColorLED [] leds = EDemoBoard.getInstance().getLEDs();

    // The value for any particular SPOT may vary by as much as 10%. For more accurate results
    // each SPOT can be calibrated to determine the zero offset and conversion factor for each axis.

    /**
     * Simple accelerometer demo to measure the tilt of the SPOT.
     * Tilt is displayed by lighting LEDs like a bubble in a level.
     */
    public void demoBubbleLevel() {
        for (int i = 0; i < 8; i++) {
            leds[i].setOff();			// turn off all LEDs
            leds[i].setColor(LEDColor.BLUE);    // set them to be blue when lit
        }
        
        while (true) {
            try {
                int tiltX = (int)Math.toDegrees(accel.getTiltX()); // returns [-90, +90]
                int offset = -tiltX / 15;       // convert angle to range [3, -3] - bubble goes to higher side
                if (offset < -3) offset = -3;
                if (offset > 3)  offset =  3;
                leds[3 + offset].setOn();       // use 2 LEDs to display "bubble""
                leds[4 + offset].setOn();
                Utils.sleep(50);                // update 20 times per second
                leds[3 + offset].setOff();      // clear display
                leds[4 + offset].setOff();
            } catch (IOException ex) {
                System.out.println("Error reading accelerometer: " + ex);
            }
        }
    }

    /*
     * If we wanted to implement our own version of getTiltX() that returns the
     * tilt value in degrees instead of radians, then here's one way to do so
     * based on the code in the LIS3L02AQAccelerometer class:
     *
     *   public int getTiltXDegrees() {
     *       double x = getAccelX();        // get current acceleration along each axis
     *       double y = getAccelY();
     *       double z = getAccelZ();
     *       double a = Math.sqrt(x*x + y*y + z*z);     // acceleration magnitude
     *       double tilt = x / a;                       // normalize
     *       double tiltRadians = MathUtils.asin(tilt); // tilt angle in radians
     *       return (int)(Math.toDegrees(tiltRadians) + 0.5);   // rounded result in degrees
     *   }
     *
     */
    
    
    /**
     * MIDlet call to start our application.
     */
    protected void startApp() throws MIDletStateChangeException {
        demoBubbleLevel();
    }

    protected void pauseApp() {
        // This will never be called by the Squawk VM
    }

    /**
     * Called if the MIDlet is terminated by the system.
     * I.e. if startApp throws any exception other than MIDletStateChangeException,
     * if the isolate running the MIDlet is killed with Isolate.exit(), or
     * if VM.stopVM() is called.
     * 
     * It is not called if MIDlet.notifyDestroyed() was called.
     *
     * @param unconditional If true when this method is called, the MIDlet must
     *    cleanup and release all resources. If false the MIDlet may throw
     *    MIDletStateChangeException  to indicate it does not want to be destroyed
     *    at this time.
     */
    protected void destroyApp(boolean unconditional) throws MIDletStateChangeException {
        for (int i = 0; i < 8; i++) {
            leds[i].setOff();
        }
    }
}
