/*
 * Copyright (c) 2006, 2007 Sun Microsystems, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 **/

package org.sunspotworld.demo;

/*
* GPIOToneGeneratorSampleCode.java
*
* Illustrates use of the GPIO pin coupled with special driver softrware
* to implement a tone generator.
*    
* For this program, you can use almost any speaker and connect
* one wire somehow to pin D0 and the other to ground pin on the SPOT demo sensor board.
* This program will make it play a major scale, repeated indefinitely.
*
* This sample illustrates how
* the programmer "binds" special drivers such as the tone generator or the 
* servo controller software to particular pins.
* To use these services, bind the proper service to the desired pin,
* and capture the returned object.
* That object has the methods you need to interact with the pin
* as appropriate for that service.
* In this case we have a tone generator. See initAndRun() below.
*
* author: Randy Smith, Steve Uhler
* date: August 2, 2006
*/

import com.sun.spot.sensorboard.EDemoBoard;
import com.sun.spot.sensorboard.peripheral.ToneGenerator;
import com.sun.spot.util.Utils;
import java.io.IOException;

import javax.microedition.midlet.MIDlet;
import javax.microedition.midlet.MIDletStateChangeException;

public class GPIOToneGeneratorSampleCode extends MIDlet {
    
    /*
     * Convert note numbers into periods
     * to feed the eDemo board tone generator.
     *
     *
     * If we choose A=440 as the base frequency of an equal
     * tempered scale, and let A=110 (two octaves down) be
     * note "0" (the lowest note), then:
     *  f = 110 * 2 ^ (n/12)
     * Where "f" is the note frequency in hz, and "n" is the note.
     *
     * The clock rate of the eDemo board is 8mhz, with the tone
     * clock set to increment every 8 cycles, which allows
     * 1,000,000 signal transitions/second.  Since it takes
     * 2 signal transitions/cycle, the period which is fed to
     * the tone generator is 500,000/f.
     *
     * As the eDemo cpu load increases with increasing frequency,
     * the minimum allowable period is "129" or about 3876hz.
     *
     * "tone_map" contains the periods for the notes in the lowest
     * octave.
     */ 
    static int tone_map[] = {
        4545, 4290, 4050, 3822, 3608, 3405, 3214, 3034, 2863, 2703, 2551, 2408
    };
    
    /*
     * These note values, when added to any starting note,
     * make up a major scale
     */ 
    static int major[] = { 0, 2, 4, 5, 7, 9, 11, 12 };
    
    EDemoBoard demo;
    ToneGenerator toneGen;
    
    /**
     * Main method. 
     **/
    private void initAndRun()throws IOException {
        
        // Instantiate a tone generator by binding the driver. We use pin D0. 
        demo = EDemoBoard.getInstance();
        toneGen = new ToneGenerator(demo.getIOPins()[EDemoBoard.D0]);
        // Play a major scale
        int start_note = 21;	// arbitrary
        while(true){
            for (int i=0;i<major.length;i++) {
                playNote(toneGen, major[i] + start_note, 500, 80);
            }
        }
    }
    
    /**
     * Calculate the period of a note.
     **/
    public static int note2period(int note) {
        // The tone_map[] only covers the lowest octave, so account for
        // multiple octaves by dividing by the appropriate power of 2.
        int octave = note/12;
        return tone_map[note%12] / (1<<octave);
    }
    
    /**
     * Play a note:
     * @param tone	The tone generator to use
     * @param note	The note number (0=A110)
     * @param dur	The total note duration in ms
     * @param len	The % of the total duration the note is playing
     */
    void playNote(ToneGenerator tone, int note, int dur, int len) {
        int on = dur * len/100; 
        tone.setPeriod(note2period(note));
        tone.setDuration(on);
        tone.beep();   // Starts an asynchronous tone (plays concurrently). Otherwise see toneOn() and toneOff()
        Utils.sleep(dur);
    }
    
    /**
     * The rest is boiler plate code, for Java ME compliance
     *
     * startApp() is the MIDlet call that starts the application.
     */
    protected void startApp() throws MIDletStateChangeException {
        try {
            initAndRun();
        } catch (IOException ex) { //A problem in reading the sensors.
            ex.printStackTrace();
        }
    }
    
    /**
     * This will never be called by the Squawk VM.
     */
    protected void pauseApp() {
    }
    
    /**
     * Called if the MIDlet is terminated by the system.
     * I.e. if startApp throws any exception other than MIDletStateChangeException,
     * if the isolate running the MIDlet is killed with Isolate.exit(), or
     * if VM.stopVM() is called.
     * 
     * It is not called if MIDlet.notifyDestroyed() was called.
     *
     * @param unconditional If true when this method is called, the MIDlet must
     *    cleanup and release all resources. If false the MIDlet may throw
     *    MIDletStateChangeException  to indicate it does not want to be destroyed
     *    at this time.
     */
    protected void destroyApp(boolean unconditional) throws MIDletStateChangeException {
    }
}
