/*
* Copyright (c) 2006, 2007 Sun Microsystems, Inc.
* 
* Permission is hereby granted, free of charge, to any person obtaining a copy
* of this software and associated documentation files (the "Software"), to 
* deal in the Software without restriction, including without limitation the 
* rights to use, copy, modify, merge, publish, distribute, sublicense, and/or 
* sell copies of the Software, and to permit persons to whom the Software is 
* furnished to do so, subject to the following conditions:
* 
* The above copyright notice and this permission notice shall be included in 
* all copies or substantial portions of the Software.
* 
* THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
* IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
* FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
* AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
* LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
* FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
* DEALINGS IN THE SOFTWARE.
 **/

package org.sunspotworld.demo;

/*
 * LightSensorSampleCode.java
 *
 * Illustrates use of the on board light Sensor. 
 *    This app prints out the light level reading.
 *    It also registers itself to be notified if the light sensor reading
 *    ever drops to zero.
 *
 *    You can find the light sensor near SW1 on the top of the Sun SPOT
 *    demo board. 
 * author: Randy Smith  
 * date: August 2, 2006 
 */ 

import com.sun.spot.sensorboard.EDemoBoard; 
import com.sun.spot.sensorboard.peripheral.ILightSensor;  
import com.sun.spot.sensorboard.peripheral.ILightSensorThresholdListener;
import com.sun.spot.util.Utils;
import java.io.IOException;

import javax.microedition.midlet.MIDlet;
import javax.microedition.midlet.MIDletStateChangeException;
 
public class LightSensorSampleCode extends MIDlet implements ILightSensorThresholdListener {   

    private ILightSensor lightSensor = EDemoBoard.getInstance().getLightSensor();   

    /**
     * Main application loop.
     * IOException may be thrown by reading the accelerometer, temperature, or light sensor.
     */
    private void run()throws IOException {   
        
        lightSensor.addILightSensorThresholdListener(this); // register us as a listener
        lightSensor.setThresholds(0, 700);                  // notify if no light measured or if really bright
        lightSensor.enableThresholdEvents(true);            // turn on notification
        
        while(true){ 
            int lightLevel = lightSensor.getValue();       // Ranges from 0 - 740
            // If you run the project from the host with the USB connected, you will see this text output...
            System.out.println("LightSensor.getValue() = " + lightLevel);;    
            Utils.sleep(200);                              // Like Thread.sleep() without the exception.
        }
    }

    /**
     * Callback when low or high light threshold is crossed.
     *
     * @param light the light sensor
     * @param val current light sensor value
     */
    public void thresholdExceeded(ILightSensor light, int val) {
        String which = (val < 100) ? "Low" : "Bright";
        System.out.println(which + " light level event: light level = " + val);
        Utils.sleep(1000);                                  // wait one second
        lightSensor.enableThresholdEvents(true);            // re-enable notification
    }
   
    /**
     * Callback when anyone changes the light sensor thresholds.
     *
     * @param light the light sensor
     * @param low cause a threshold event when light sensor value is less than or equal low
     * @param high cause a threshold event when light sensor value is greater than or equal high
     */
    public void thresholdChanged(ILightSensor light, int low, int high) {
        // ignore 
    }
    
    /**
     * The rest is boiler plate code, for Java ME compliance
     *
     * startApp() is the MIDlet call that starts the application.
     */
    protected void startApp() throws MIDletStateChangeException { 
        try {
            run();
        } catch (IOException ex) { //A problem in reading the sensors. 
            ex.printStackTrace();
        }
    }

    /**
     * This will never be called by the Squawk VM.
     */
    protected void pauseApp() { 
    }

    /**
     * Called if the MIDlet is terminated by the system.
     * I.e. if startApp throws any exception other than MIDletStateChangeException,
     * if the isolate running the MIDlet is killed with Isolate.exit(), or
     * if VM.stopVM() is called.
     * 
     * It is not called if MIDlet.notifyDestroyed() was called.
     *
     * @param unconditional If true when this method is called, the MIDlet must
     *    cleanup and release all resources. If false the MIDlet may throw
     *    MIDletStateChangeException  to indicate it does not want to be destroyed
     *    at this time.
     */
    protected void destroyApp(boolean unconditional) throws MIDletStateChangeException {
    }

}
