/*
 * SensorChecker.java
 *
 * Created on September 13, 2007, 8:24 PM
 *
 */

package org.sunspotworld;

import com.sun.spot.sensorboard.EDemoBoard;
import com.sun.spot.sensorboard.io.IIOPin;
import com.sun.spot.sensorboard.io.IInputPin;
import com.sun.spot.sensorboard.io.IInputPinListener;
import com.sun.spot.sensorboard.io.IOutputPin;
import com.sun.spot.sensorboard.io.IScalarInput;
import com.sun.spot.sensorboard.peripheral.IAccelerometer3D;
import com.sun.spot.sensorboard.peripheral.ILightSensor;
import com.sun.spot.sensorboard.peripheral.ILightSensorThresholdListener;
import com.sun.spot.sensorboard.peripheral.ISwitch;
import com.sun.spot.sensorboard.peripheral.ISwitchListener;
import com.sun.spot.sensorboard.peripheral.ITemperatureInput;
import com.sun.spot.sensorboard.peripheral.ITriColorLED;
import com.sun.spot.sensorboard.peripheral.LEDColor;
import com.sun.spot.sensorboard.peripheral.LIS3L02AQAccelerometer;
import com.sun.spot.util.Utils;
import java.io.IOException;
import javax.microedition.midlet.MIDlet;
import javax.microedition.midlet.MIDletStateChangeException;

/**
 * This MIDlet demonstrates how the Virtual Sensor Panel can be used 
 * in the SPOT World Emulator to interact with a virtual SPOT.
 * The LEDs on the virtual SPOT display the value read from one
 * of the SPOT's sensors.
 *
 * There are four different modes:
 *
 * 1. Display the light sensor reading in white
 * 2. Display the temperature sensor reading in red.
 * 3. Display the analog input A0 in green.
 * 4. Display the Z acceleration in blue.
 *
 * Pushing the left switch (SW1) advances to the next mode.
 * The current mode is shown by setting one of H0-H3 to high.
 *
 * Also D0 is set as an output and the application sets it
 * to mirror the value that D1 is set to.
 *
 * @author Ron Goldman
 */
public class SensorChecker extends MIDlet implements ILightSensorThresholdListener, ISwitchListener, IInputPinListener {
    
    private EDemoBoard demo = EDemoBoard.getInstance();
    private ILightSensor light = demo.getLightSensor();
    private ITemperatureInput temp = demo.getADCTemperature();
    private IScalarInput scalars[] = demo.getScalarInputs();
    private ISwitch sw[] = demo.getSwitches();
    private ITriColorLED leds[] = demo.getLEDs();
    private IOutputPin outs[] = demo.getOutputPins();
    private IIOPin ins[] = demo.getIOPins();
    private LIS3L02AQAccelerometer accel = (LIS3L02AQAccelerometer)demo.getAccelerometer();
    private LEDColor colors[] = { LEDColor.WHITE, LEDColor.RED, LEDColor.GREEN, LEDColor.BLUE };
    private int mode = 0;
    
    protected void startApp() throws MIDletStateChangeException {
        sw[0].addISwitchListener(this);
        light.addILightSensorThresholdListener(this);
        light.setThresholds(10, 740);
        light.enableThresholdEvents(true);
        outs[mode].setHigh();
        ins[0].setAsOutput(true);
        ins[1].addIInputPinListener(this);
        accel.setScale(LIS3L02AQAccelerometer.SCALE_6G);
        while (true) {
            try {
                int bars = 0;
                switch (mode) {
                    case 0:
                        bars = light.getValue() / 84;     // [0, 750] = [0, 8]
                        break;
                    case 1:
                        bars = (int)(temp.getFahrenheit() + 40.0) / 20;
                        break;
                    case 2:
                        bars = scalars[0].getValue() / 114;
                        break;
                    case 3:
                        double accZ = accel.getAccelZ();
                        // System.out.println("Z acceleration = " + accZ);
                        bars = (int)((accZ + 6) / 1.333);
                        break;
                }
                for (int i = 0; i < 8; i++) {
                    leds[i].setColor(colors[mode]);
                    leds[i].setOn(i < bars);
                }
                Utils.sleep(200);
            } catch (IOException ex) {
                
            }
        }
    }
    
    protected void pauseApp() {
        // This will never be called by the Squawk VM
    }
    
    protected void destroyApp(boolean arg0) throws MIDletStateChangeException {
        // Only called if startApp throws any exception other than MIDletStateChangeException
    }

    /**
     * Callback for when the light sensor value goes above or below 
     * the specified thresholds.
     * 
     * @param light the ILightSensor that has crossed a threshold.
     * @param val the current light sensor reading.
     */
    public void thresholdExceeded(ILightSensor light, int val) {
        System.out.println("Light threshold exceeded: " + val);
        Utils.sleep(2000);
        light.enableThresholdEvents(true);
    }

    /**
     * Callback for when the light sensor thresholds are changed.
     * 
     * @param light the ILightSensor that had its threshold values changed.
     * @param low the new light sensor low threshold value.
     * @param high the new light sensor high threshold value.
     */
    public void thresholdChanged(ILightSensor light, int low, int high) {
    }

    public void switchReleased(ISwitch sw) {
        outs[mode].setHigh(false);
        if (++mode > 3) { 
            mode = 0;
        }
        outs[mode].setHigh(true);
    }

    public void switchPressed(ISwitch sw) {
    }

    public void pinSetLow(IInputPin pin) {
        ins[0].setLow();
    }

    public void pinSetHigh(IInputPin pin) {
        ins[0].setHigh();
    }
}
