import os, datetime, email
from email import Utils
from eventgen import *
from appeventgen import *

class EventGenerator:
    def __init__(self):
        self.events = []
        
    def hasNext(self):
        if len(self.events) > 0:
            return True
        return False
    
    def peekNextEvent(self):
        if len(self.events) < 1:
            return None
        return self.events[0].date
        
    def getNextEvent(self):
        ev = self.events[0]
        del self.events[0]
        return ev
    
    def insertEvent(self, event):
        #print "INSERT"
        #print "new event ", event.date
        #print "before ", self.events
        if len(self.events) < 1:
            self.events.append(event)
            return 
        i = 0
        while (i < len(self.events)) and self.events[i].date < event.date:
            i += 1
        if i == len(self.events):
            self.events.append(event)
        else:
           self.events.insert(i, event)  
        #print "after ", self.events

class SimEventGenerator(EventGenerator):
    pass

class CommEventGenerator(EventGenerator):
    pass

class DeviceEventGenerator:
    "The EventGenerator is an iterator used to iterate over all events in the trace."
    def __init__(self, fname):
        self.nodes, self.appdatanodes = self.processFiles(fname)

    def processFiles(self, dirname):
        """
        Called by the constructor to read in trace data.  Data is expected to be 'consolidated'.
        Data should be in a data subdirectory of the directory where the program is run.
        """
        nodes = []      
        appdatanodes = {}      
        for root, dirs, files in os.walk(dirname):
            for name in files:                                
                totalonline = 0      
                lastdate = None
                dates = []
                fname = dirname + "/" + name
                id = name.split(".")[0]
                file = open(fname, "r")
                n = DeviceTraceData(id)
                lastmeasurement = None
                laststatechange = None
                while 1:
                    #date
                    date = file.readline()

                    if not date: break
                    #battery
                    bat = file.readline()
                    bat = bat.split()[1]
                    if not bat: break
                    
                    #onAC
                    onac = file.readline().split()[1]
                    if not onac: break
                    
                    #cpu
                    cpu = file.readline()
                    if not cpu: break
                    
                    #disk
                    disk = file.readline()
                    if not disk: break
                    
                    #connected
                    conn = file.readline().split()[1]
                    if not conn: break
                    
                    #idle time
                    idle = file.readline()
                    if not idle: break
                    
                    thismeasurement = Measurement(date, bat, onac, conn)
                    if thismeasurement.conn == '1':
                        totalonline += 1
                        dates.append(datetime.datetime.fromtimestamp(email.Utils.mktime_tz(email.Utils.parsedate_tz(date))))

                    if(lastmeasurement != None):
                        statechange = lastmeasurement.stateChange(thismeasurement, laststatechange)
                    else:
                        statechange = StateChange()
                        statechange.setVals(thismeasurement)
                        
                    if(statechange != None):
                        n.events.append(statechange)
                        laststatechange = statechange


                    lastmeasurement = thismeasurement
                    #save last date for turning off at end of run        
                    lastdate = laststatechange.date
                #add one last off event at the end of the run
                appdatanodes[id] = AppData(dates, totalonline)
                n.events.append(StateChange((lastdate+datetime.timedelta(0, 500, 0)), True, False, False, False, 100, True))
                nodes.append(n)
                        
            return nodes, appdatanodes

    def hasNext(self):
        """
        Returns True if there is another event and False otherwise.
        """
        for n in self.nodes:
            if len(n.events) > 0:
                return True
        return False
            
    def peekNextEvent(self):
        mintime = -1
        minindex = 0
        for n in range(len(self.nodes)):
            if (mintime == -1 and len(self.nodes[n].events) > 0) or (len(self.nodes[n].events) > 0 and self.nodes[n].events[0].date < mintime):
                mintime = self.nodes[n].events[0].date
                minindex = n
        if mintime == -1:
            return None
        return mintime
            
    def getNextEvent(self):
        """
        Returns the next Event object.
        """
        mintime = -1
        minindex = 0
        for n in range(len(self.nodes)):
            if (mintime == -1 and len(self.nodes[n].events) > 0) or (len(self.nodes[n].events) > 0 and self.nodes[n].events[0].date < mintime):
                mintime = self.nodes[n].events[0].date
                minindex = n
        if mintime == -1:
            return None
    
        minevent = self.nodes[minindex].events[0]
        del self.nodes[minindex].events[0]       
        stateinfo =  DeviceStateInfo(minevent.ison, minevent.bat, minevent.onac)
        changeinfo = DeviceChangeInfo(minevent.onchange, minevent.batchange, minevent.onacchange)
        return DeviceEvent(self.nodes[minindex].id, minevent.date, stateinfo, changeinfo)

class AppEventGenerator(EventGenerator):
    def __init__(self, appdatanodes):
        EventGenerator.__init__(self)
        keys = appdatanodes.keys()
        
        for key in keys:
            newevs = poisson((.5/6), appdatanodes[key].totalonline)
            for i in range(len(newevs)):
                for j in range(newevs[i]):
                    num = random.randint(0, len(keys)-1)
                    while key == keys[num]:
                        num = random.randint(0, len(keys)-1)                        
                    event = AppEvent(key, (appdatanodes[key].dates[i]), key, keys[num], [], ORIG)
                    self.insertEvent(event)
        print "LENGTH ", len(self.events)


class QueueManager:
    def __init__(self, deq, ceq, seq, aeq=None):
        self.mainq = []
        self.mainq.append(deq)
        self.mainq.append(ceq)
        self.mainq.append(seq)
        self.mainq.append(aeq)
    
    def hasNext(self):
        for q in self.mainq:
            if isinstance(q, DeviceEventGenerator) and q.hasNext() == False:
                print "NO MORE DEVICE EVENTS"
                for i in self.mainq:
                    if isinstance(i, AppEventGenerator):
                        print "LENGTH OF APP QUEUE ", len(i.events)
                #if there are no more device events, return False
                #FIXME - print state of appevent queue
                return False
            if q.hasNext():
                return True
        return False
    
    def getNextEvent(self):
        mintime = None
        minq = None
        for q in self.mainq:
            #print "q ", q
            #print "mintime ", mintime
            #print "next event ", q.peekNextEvent()
            if mintime == None or (q != None and q.hasNext() and q.peekNextEvent() < mintime):
                mintime = q.peekNextEvent()
                minq = q
        if minq == None:
            return minq
        return minq.getNextEvent()     