"""
device_utils.py

This file contains the classes relevant to DeviceEvents
and creating/maintaining information about the device.

Author: Sami Rollins
"""

from event_generation_utils import Event, EventGenerator

class DeviceEvent(Event):
    """
    DeviceEvent inherits from Event and 
    represents an event at the device level, eg:
      device turns on/off
      device unplugs/plugs in
      battery level goes up/down
    """
    def __init__(self, id, date, stateinfo, changeinfo):
        """
        DeviceEvent inherits from Event and also 
        maintains info about the state of the device
        after the event is applied and the 
        components of the device that will change 
        state as a result of this event.
        """
        Event.__init__(self, id, date)
        self.stateinfo = stateinfo
        self.changeinfo = changeinfo
    
    def __str__(self):
        superstr = Event.__str__(self)
        statestr = self.stateinfo.__str__()
        changestr = self.changeinfo.__str__()
        return ("Device " + superstr + statestr + changestr)

class DeviceStateInfo:
    """
    A DeviceStateInfo object maintains the following
    information about the state of a device:
    ison - True if the device is on and connected to a network
    bat - integer value representing the battery level
    onac - True if the device is plugged in to a power outlet
    """
    def __init__(self, ison=0, bat=0, onac=0):
        self.ison = ison
        self.bat = bat
        self.onac = onac
    def __str__(self):
        returnstr = "\n\tON: " + str(self.ison) + \
                    "\n\tBAT: " + str(self.bat) + \
                    "\n\tAC: " + str(self.onac)  
        return returnstr                     

class DeviceChangeInfo:
    """
    A DeviceChangeInfo object identifies the state components
    that have changed as a result of a given event:
    onchange - True if the device went from on->off or off->on
    batchange - True of the battery level changed
    onacchange - True if the device plugged in or unplugged
    """
    def __init__(self, onchange=0, batchange=0, onacchange=0):
        self.onchange = onchange
        self.batchange = batchange
        self.onacchange = onacchange
    def __str__(self):
        returnstr = "\n\tONCHANGE: " + str(self.onchange) + \
                    "\n\tBATCHANGE: " + str(self.batchange) + \
                    "\n\tACCHANGE: " + str(self.onacchange)
        return returnstr
    
    
"""
start testing
"""
import sys, datetime

def main(argv=None):
    de = DeviceEvent(0, datetime.datetime.now(), DeviceStateInfo(True, 34, True), DeviceChangeInfo(False, True, False))
    print de
    
    
if (__name__ == "__main__"):
    sys.exit(main())

"""
end testing
"""