"""
event_generation_utils.py

This file contains the base classes for the 
event generation portion of the simulator.

Author: Sami Rollins
"""
import datetime
from heapq import heappush, heappop

class Event:
    """
    Base class for all simulator Events.
    All events have the following attributes:
    id - the device on which the event occurs 
         current implementation uses the mac
         address derived from the file name
    date - datetime object to represent
           the time at which the event occurs
    """
    
    def __init__(self, id, date):
        self.id = id
        self.date = date

    def __str__(self):
        returnstr =  "Event " + \
                        "\tID: " + str(self.id) + \
                        "\tTime: " + str(self.date)
        return returnstr

    """
    Methods for comparing two Event objects.
    """
    def __lt__(self, other):
        return self.date < other.date
        
    def __gt__(self, other):
        return self.date > other.date

    def __le__(self, other):
        return self.date <= other.date
        
    def __ge__(self, other):
        return self.date >= other.date



class EventGenerator:
    """
    The EventGenerator acts as an priority queue
    for events.  Events can be inserted into an
    EventGenerator.  When events are removed, the
    most recent event is returned.
    """
    def __init__(self):
        self.events = []
        
    def hasNext(self):
        if len(self.events) > 0:
            return True
        return False
    
    def peekNextEvent(self):
        if len(self.events) < 1:
            return None
        #property of heapq is that
        #the smallest element is at
        #position 0
        return self.events[0].date
        
    def getNextEvent(self):
        return heappop(self.events)
    
    def insertEvent(self, event):
        heappush(self.events, event)

"""
start testing
"""
import sys, time, random

def main(argv=None):
    eg = EventGenerator()
    list = []
    #create list of events
    for i in range(5):
        e = Event((i*random.randint(0, 235)), datetime.datetime.now())
        list.append(e)
        time.sleep(1)
    #randomize list of events
    for i in range(5):
        randnum = random.randint(0, 4)
        tmp = list[i]
        list[i] = list[randnum]
        list[randnum] = tmp
    #insert random events into generator queue
    for i in list:    
        print "inserting ", i
        eg.insertEvent(i)
    #retrieve events from queue
    while eg.hasNext(): 
        print eg.peekNextEvent()
        print eg.getNextEvent()
        print "***********"
   
if (__name__ == "__main__"):
    sys.exit(main())

"""
end testing
"""
