#!/usr/bin/python

#--------------------------------------------------
# predictor.py
#
# author:    cheryl chang-yit
#            cchangyit@cs.usfca.edu
#
# NOTES:
# - Implemented as described in the Mickens [1] paper.
#  
# - Code for the linear prediction algorithm 
#   derived from WLSE adaptive-predictor MatLab code in 
#  
#      M. Garg. "Linear Prediction Algorithms (00D07015)." Indian Institute 
#      of Technology, Bombay,India. April 24, 2003.
#  
#   using additional Python modules 
#          Numeric
#
#
# REFERENCES:
# [1]  C. Chang-Yit.  search.py.  Code written for class cs662, Computer 
#      Science Department. University of San Francisco, San Francisco, CA.
#
# [2]  J. Douglas. "Adaptive Filters." Connexions. May 12, 2005. 
#      http://cnx.org/content/col10280/1.1/.
#
# [3]  M. Garg. "Linear Prediction Algorithms (00D07015)." Indian Institute 
#      of Technology, Bombay,India. April 24, 2003.
#
# [4]  J. Mickens and B. Noble. Exploiting Availability Prediction in 
#      Distributed Systems. In Proceedings of the 2006 NSDI, pages 73--86, 
#      San Jose, CA, May 2006.
#
# [5]  Unknown.  De Bruijn graph.  Wikipedia, the free encyclopedia, 
#      http://en.wikipedia.org/wiki/De_Bruijn_graph. Last modified 09:07, 
#      20 November 2006.
#
# [6]  Unknown.  De Bruijn sequence.  Wikipedia, the free encyclopedia, 
#      http://en.wikipedia.org/wiki/De_Bruijn_sequence. Last modified 21:13, 
#      8 April 2007.
#
#--------------------------------------------------


import math, time
from random import randrange   #used in predictors Random, History, Hybrid

#for SatX, History, TwiddledHistory predictors and Hybrid predictors
from counter import SaturatingCounter

#for History and TwiddledHistory predictors
from deBruijn import GraphDeBruijnNode
from deBruijn import GraphDeBruijn

#for linear predictor only
from Numeric import *

##############################
### GLOBAL DEFINITIONS
###
_TIMEUNIT_MINUTE = 1
_TIMEUNIT_HOUR = 2


### GLOBAL DATA STRUCTURES
_RANDOM_DISALLOWED = False

### GLOBAL METHODS
def disableRandomBehavior():
    """Disable all random predictor behavior for debugging purposes."""
    _RANDOM_DISALLOWED = True

def enableRandomBehavior():
    """Enable random predictor behavior for more fairly distributed results."""

    ###TODO  .........implement this for History predictor type!!!
    _RANDOM_DISALLOWED = False

########################################
# Predictor superclass
#
class Predictor(object):
    """The abstract Predictor superclass.  The discrete range of prediction values is 
set {0, 1}.  DO NOT use this class directly!"""
    #
    #   An individual prediction has a specific time for which it predicts its 
    #   value will hold.  An individual prediction is mutable in that it may be 
    #   updated if the predictor receives new information
    #
    #   prediction =  [{'time':time_stamp, 'prediction':value}]
    #

    prediction_range = (0, 1)  #immutable

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0):
        super(Predictor, self).__init__(lookahead_max, history_max, prediction_init)
        self.history_count = history_max
        self.lookahead_count = lookahead_max
        self.history = []
        self.predictions = []

        #for pickling trained predictor state
        self.current_time = 0

        #for pickling backed up predictor state
        self.bkp_last_processed_timestamp = None   ##GMT

    def deleteObsoletePredictions(self, current_time):
        ###figure out how many predictions to shift out, assume sorted by time
        count = 0
        for p in self.predictions:
            if p['time'] <= current_time:
                count += 1
        for i in xrange(0,count):
            self.predictions.pop(0)

        return count

    def getCurrentTime(self):
        return self.current_time

    def update(self, current_time, value):
        """Update predictions based on value observed at the current time.  
Shift out any predictions earlier than the current time."""
        pass

    def makePrediction(self, time):
        """Return a prediction for the specified time."""
        #make a prediction for the specified time by retrieving from queue.
        #assume predictions inserted in order of time => sorted Q, no duplicates

        #prediction of form {'time':##, 'prediction':##}
        prediction = None
        for p in self.predictions:
            if self.isTimeMatch(p['time'], time):
                prediction = p['prediction']
            elif p['time'] > time:
                break

        return prediction

    def isTimeMatch(self, time_id, time_try):
        is_match = False
        if time_id == time_try:    #prediction for *exactly* time or none
            is_match = True

        return is_match

    def getLookahead(self):
        return self.lookahead_count

    def getHistoryLength(self):
        return self.history_count

    def getHistory(self):
        return self.history

    def printHistory(self):
        print self.history

    ####################
    ### timestamp-related for conversions with time-unit intervals
    ###
    def setTimestampStart(self, tstamp):
        """Sets timestamp corresponding to time-unit t=0"""
        self.timestamp_start = tstamp

    def setTimestampIntervalAsMinutes(self, interval):
        """Sets length of time interval to equivalent of specified minutes"""
        self.timestamp_interval = interval

    def setTimestampIntervalAsHours(self, interval):
        """Sets length of time interval to equivalent of specified hours"""
        self.timestamp_interval = interval * 60

    def getTimestampStart(self):
        return self.timestamp_start

    def getTimestampIntervalAsMinutes(self):
        return self.timestamp_interval

    def getTimestampIntervalAsHours(self):
        return self.timestamp_interval / 60.0

########################################
# PredictorRandom class
#
class PredictorRandom (Predictor):
    """Predictor that makes irreproducible predictions at random.  It keeps 
no history and no lookahead prediction queue.  Every prediction is randomized 
anew, including repeated requests for the same future time."""

    def __init__(self, lookahead_max):
        history_max = 0
        prediction_init = None
        super(PredictorRandom, self).__init__(lookahead_max, history_max, prediction_init)

        ###no state held
        pass

    def update(self, current_time, current_value):
        self.current_time = current_time
        pass

    def makePrediction(self, time):
        prediction = randrange(0, 2)  #integer 0 or 1

        return prediction

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max:           ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "None"
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "None"
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorAlways class
#
class PredictorAlways (Predictor):
    """Predictor that always makes the same prediction, whatever prediction 
value is set upon creation of the predictor."""

    def __init__(self, lookahead_max=0, prediction_init=0):
        history_max = 0
        super(PredictorAlways, self).__init__(lookahead_max, history_max, prediction_init)

        ###one bit of unchanging state held, the starting state
        self.next_prediction = prediction_init

        ###initialize predictions
        for n in xrange(1,lookahead_max+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def update(self, current_time, current_value):

        self.current_time = current_time
        last_time = self.predictions[-1]['time']

        ###get rid of outdated predictions
        deleted_count = self.deleteObsoletePredictions(current_time)

        ###update old predictions as necessary
        for p in self.predictions:
            p['prediction'] = self.next_prediction

        ###generate enough new predictions to fill out queue
        for i in xrange(0,deleted_count):
            prediction =  {'time': last_time+1+i, 'prediction':self.next_prediction}
            self.predictions.append(prediction)

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max:           ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "None"
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "None"
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorRightNow class
#
class PredictorRightNow (Predictor):
    """Predictor that predicts the current state."""

    def __init__(self, lookahead_max=0, prediction_init=0):
        history_max = 0
        super(PredictorRightNow, self).__init__(lookahead_max, history_max, prediction_init)

        ###no state held
        pass

        ###initialize predictions
        for n in xrange(1,lookahead_max+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def update(self, current_time, current_value):

        self.current_time = current_time
        last_time = self.predictions[-1]['time']

        ###get rid of outdated predictions
        deleted_count = self.deleteObsoletePredictions(current_time)

        ###update old predictions as necessary
        for p in self.predictions:
            p['prediction'] = current_value

        ###generate enough new predictions to fill out queue
        for i in xrange(0,deleted_count):
            prediction =  {'time': last_time+1+i, 'prediction':current_value}
            self.predictions.append(prediction)

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max:           ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "None"
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "None"
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorSatX class
#
class PredictorSatX (Predictor):
    """Predictor that predicts based on the current state, weighted by a X bits 
of history."""

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, X=1):
        super(PredictorSatX, self).__init__(lookahead_max, history_max, prediction_init)

        ###X bits of state held
        ###      X=2  =>  a 2-bit saturating counter, 
        ###               able to assume four values {-2, -1, +1, +2}

        self.X = X
        self.satX = SaturatingCounter(X)

        #set initial saturation counter value
        self.prediction_init = prediction_init

        ###initialize predictions
        for n in xrange(1,lookahead_max+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def update(self, current_time, current_value):

        self.current_time = current_time
        last_time = self.predictions[-1]['time']

        ###get rid of outdated predictions
        deleted_count = self.deleteObsoletePredictions(current_time)

        ###update old predictions as necessary
        if current_value <= 0:   #0 or negative => offline
            self.satX.decrement()
        elif current_value > 0:  #positive => online
            self.satX.increment()

        for p in self.predictions:
            p['prediction'] = self.satcounterToPrediction(self.satX)

        ###generate enough new predictions to fill out queue
        for i in xrange(0,deleted_count):
            prediction =  {'time': last_time+1+i, 'prediction':self.satcounterToPrediction(self.satX)}
            self.predictions.append(prediction)

    def satcounterToPrediction(self,sat_counter):
        prediction = None #this reflects an error
        if sat_counter.getValue() < 0:  #negative counter
            prediction = 0
        else:                     #positive counter
            prediction = 1
        return prediction

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max:           ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "SatX X:                ", self.X
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "SatX Counter:", self.satX.getValue()
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorHistory class
#
class PredictorHistory (Predictor):
    """Predictor that predicts availability based on the most recent uptime state sequence."""

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, X=1):
        super(PredictorHistory, self).__init__(lookahead_max, history_max, prediction_init)

        self.history_count = history_max
        self.lookahead_count = lookahead_max

        ###observed uptime state sequences held in a de Bruijn graph
        ###    in which each node contains a saturating counter and 
        ###    where X = the max range of the counter

        ###graph building
        ###optimize graph node creation to create nodes only as needed.  
        ###assigning data to a node creates it
        self.alphabet = ['0','1']  ##limited by SatX counter binary predictions
        self.X = X
        self.graph = GraphDeBruijn(self.alphabet, self.history_count)

        ###initialize uptime history sequence
        self.uptime_sequence = ""
        for i in xrange(0,self.history_count):
            self.uptime_sequence += str(prediction_init)

        ###initialize predictions
        for n in xrange(1,self.lookahead_count+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def update(self, current_time, current_value):

        self.current_time = current_time

        ###update the last uptime sequence node counter
        if (current_value > 0):  #availabile
            self.incrementCounter(self.uptime_sequence)
        else:  #not available
            self.decrementCounter(self.uptime_sequence)

        ###update the uptime sequence with the current availability value
        self.uptime_sequence = self.uptime_sequence[1:] + str(current_value)

        ###regenerate all predictions to fill queue
        seq = self.uptime_sequence
        self.predictions = []
        for i in xrange(current_time+1,current_time+1+self.lookahead_count):
            most_likely_pred = self.generateNextPrediction(seq)

            prediction =  {'time': i, 'prediction':most_likely_pred}
            self.predictions.append(prediction)
            seq = seq[1:] + str(most_likely_pred)

    def generateNextPrediction(self,sequence):
        prediction = None
        next_sequences = self.graph.getNextSequences(sequence)

        ###choose most likely next symbol
        most_likely_seq = self.chooseMostLikely(next_sequences)
        return int(most_likely_seq[-1:])

    def chooseMostLikely(self,choices):

        ###sort nodes into lists of similar satcount magnitudes
        satcount_magnitudes = {}
        for mag in xrange(0, self.X+1):
            satcount_magnitudes[mag] = []

        for seq in choices:
            satX_mag = int(math.fabs(self.getSequenceCounter(seq).getValue()))
            satcount_magnitudes[satX_mag].append(seq)

        ###from set of nodes of highest saturation magnitude, pick randomly
        ###     since they are all be equally likely
        most_likely_seq = None
        highest_mag = self.X
        while ((most_likely_seq == None) & (highest_mag >= 0)):
            likely_seqs = satcount_magnitudes[highest_mag]
            if (len(likely_seqs) > 0):

                if (globals()['_RANDOM_DISALLOWED']):  ##FOR UNIT TESTING
                    ## sort and pick 1st sequence, ascending order
                    sorted_seqs = likely_seqs.keys()
                    sorted_seqs.sort()
                    most_likely_seq = sorted_seqs[0]
                else:  ## NORMAL OPERATIONS
                    most_likely_seq = likely_seqs[randrange(0, len(likely_seqs))]

            highest_mag -= 1

        return most_likely_seq

    def getSequenceCounter(self, seq):
        counter = self.graph.getNodeData(seq)
        if (counter == None):
            self.graph.setNodeData(seq, SaturatingCounter(self.X))
            counter = self.graph.getNodeData(seq)
        return counter

    def getSequenceCounterValue(self, seq):
        return self.getSequenceCounter(seq).getValue()

    def incrementCounter(self, seq):
        self.getSequenceCounter(seq).increment()

    def decrementCounter(self, seq):
        self.getSequenceCounter(seq).decrement()

    def printGraph(self):
        """Print de Bruijn graph for debugging purposes."""
        self.graph.printSparseGraph()

    def getHistory(self):
        return self.uptime_sequence

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max (=k):       ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "DeBruijn Graph SatX X: ", self.X
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "Current Uptime Signal (k-bits): \n", self.uptime_sequence
        print ""
        print "DeBruijn Graph: \n", self.printGraph()
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorTwiddledHistory class
#
class PredictorTwiddledHistory (Predictor):
    """Predictor that predicts availability based on the most recent uptime 
state sequence and the uptime state sequences with similar saturation 
counter values."""

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, X=1, d=1):
        super(PredictorTwiddledHistory, self).__init__(lookahead_max, history_max, prediction_init)

        self.history_count = history_max
        self.lookahead_count = lookahead_max

        ###observed uptime state sequences held in a de Bruijn graph
        ###    in which each node contains a saturating counter and 
        ###    where X = the max range of the counter

        ###graph building
        self.alphabet = ['0','1']  ##limited by SatX counter binary predictions
        self.X = X
        self.d = d
        self.graph = GraphDeBruijn(self.alphabet, self.history_count)
        self.visited_sequences = []  ##helps find existing similar sequences

        ###initialize uptime history sequence
        self.uptime_sequence = ""
        for i in xrange(0,self.history_count):
            self.uptime_sequence += str(prediction_init)

        ###initialize predictions
        for n in xrange(1,self.lookahead_count+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def update(self, current_time, current_value):

        self.current_time = current_time

        ###update the last uptime sequence node counter
        if (current_value > 0):  #availabile
            self.incrementCounter(self.uptime_sequence)
        else:  #not available
            self.decrementCounter(self.uptime_sequence)
        self.visited_sequences.append(self.uptime_sequence)

        ###update the uptime sequence with the current availability value
        self.uptime_sequence = self.uptime_sequence[1:] + str(current_value)

        ###regenerate all predictions to fill queue
        seq = self.uptime_sequence
        self.predictions = []
        for i in xrange(current_time+1,current_time+1+self.lookahead_count):
            most_likely_pred = self.generateNextPrediction(seq)

            prediction =  {'time': i, 'prediction':most_likely_pred}
            self.predictions.append(prediction)
            seq = seq[1:] + str(most_likely_pred)

    def generateNextPrediction(self,sequence):
        prediction = None
        similar_sequences = self.getSimilarObservedSequences(sequence)  #including current one!

        ###for all of the similar sequences, average the saturating counter values
        sum = 0.0
        avg = 0.0
        n = len(similar_sequences)
        for ss in similar_sequences:
            sum += self.getSequenceCounterValue(ss)
        if n > 0:
            avg = sum / n

        ###predict based on average saturating counter value
        if avg > 0:  #available
            prediction = 1
        else:  #not available
            prediction = 0

        return prediction

    def getSimilarObservedSequences(self,seq):
        """Get all observed sequences differing from the given sequence by at most d symbols."""
        similar_seqs = [seq]

        for test_seq in self.visited_sequences:
            if (self.isSimilarSequence(seq, test_seq, self.d)):
                similar_seqs.append(test_seq)

        return similar_seqs

    def isSimilarSequence(self, seq, test_seq, d):
        """Return true if number of differing positions is <= d; return false otherwise"""
        isSimilar = False
        differ_count = 0
        i = 0
        while (differ_count <= d) & (i < len(seq)):
            if (seq[i:i+1] != test_seq[i:i+1]):
                differ_count += 1
            i += 1

        if (differ_count <= d):
            isSimilar = True
        return isSimilar

    def getSequenceCounter(self, seq):
        counter = self.graph.getNodeData(seq)
        if (counter == None):
            self.graph.setNodeData(seq, SaturatingCounter(self.X))
            counter = self.graph.getNodeData(seq)
        return counter

    def getSequenceCounterValue(self, seq):
        return self.getSequenceCounter(seq).getValue()

    def incrementCounter(self, seq):
        self.getSequenceCounter(seq).increment()

    def decrementCounter(self, seq):
        self.getSequenceCounter(seq).decrement()

    def printGraph(self):
        """Print de Bruijn graph for debugging purposes."""
        self.graph.printSparseGraph()

    def getHistory(self):
        return self.uptime_sequence

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max (=k):       ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "d:                     ", self.d
        print "DeBruijn Graph SatX X: ", self.X
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "Current Uptime Signal (k-bits): \n", self.uptime_sequence
        print ""
        print "Visited Sequences (k-bits): \n", self.visited_sequences
        print ""
        print "DeBruijn Graph: \n", self.printGraph()
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorLinear class
#
class PredictorLinear (Predictor):
    """Predictor that predicts availability based on the most recently 
observed values of an uptime signal."""

    ###observed uptime state held in a signal history k-bits long, 
    ###    where k=history_max
    ###
    ### Code for the linear prediction algorithm 
    ###    derived from WLSE adaptive-predictor MatLab code in 
    ###
    ###    M. Garg. "Linear Prediction Algorithms (00D07015)." Indian Institute 
    ###    of Technology, Bombay,India. April 24, 2003.
    ###

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, a=0.99):
        super(PredictorLinear, self).__init__(lookahead_max, history_max, prediction_init)

        self.history_count = history_max
        self.lookahead_count = lookahead_max

        ###initialize uptime history signal
        self.history = [prediction_init]*history_max
        self.M = history_max    #order of the predictor
        self.alpha = a          #forgetting factor

        ###initialize predictions
        for n in xrange(1,lookahead_max+1):
            prediction = {'time':n, 'prediction':prediction_init}
            self.predictions.append(prediction)

    def flipud(self, M): 
        """Returns a 2-D matrix with the columns preserved and 
        rows flipped in the up/down direction.  Only works with 2-D array. 

        Derived from numpy module function.
        """ 
        M = asarray(M) 
        if len(M.shape) != 2: 
            raise ValueError, "flipud(M), M must be a 2-D array" 
        return M[::-1] 
    
    def update(self, current_time, current_value):

        self.current_time = current_time

        ###update the uptime signal with the current observation
        self.history.append(current_value)
	#for i in self.history:
	#    print i, " ",
	#print "dropping: ", self.history.pop(0)  #drop oldest observation
	self.history.pop(0)
	#print "time to pop: ", (end-start)
	#print "size of history: ", len(self.history)

        ###regenerate all predictions to fill queue
        signal = []   ##will hold projected future signal
        signal.extend(self.history)
        self.predictions = []  #out with the old!

        for i in xrange(current_time+1,current_time+1+self.lookahead_count):
	    start = time.time()*1000
            prediction_value = self.generateNextPrediction(signal, self.M, self.alpha)
	    end = time.time()*1000
	    #print "time to generate prediction: ", (end-start)
            prediction =  {'time': i, 'prediction': prediction_value}
            self.predictions.append(prediction)

            #add prediction to end of signal and pop off oldest
            #for generating next prediction
            signal.pop(0)   #FIFO
            signal.append(prediction_value)

    def generateNextPrediction(self, signal, M, alpha):
        """Code for the WLSE adaptive-predictor
        
        The only use of these module imports:
            from Numeric import *
        """
        prediction = None

        #convert signal to vector
        x = zeros((len(signal),1), Float)
        for i in xrange(0, len(signal)):
            x[i] = signal[i]

        ### initialize adaptive linear prediction structures
        ###    - the coefficient vector a
        ###    - the P matrix
        ###    - the output prediction vector s
        a = zeros((M,1), Float)
        a[0] = 1
        P = identity(M)  #identity matrix
        s = zeros((size(x)), Float)
        s[0] = x[0]
        for k in xrange (1,len(x)): #prediction starts at t=2

            ### current input vector 
            ### get M elements, ordered newest to oldest
            currin = zeros((M,1), Float)
            if((k-M) >= 0):   #k-M is an element index 0, 1, ...
                currin=self.flipud(x[k-M:k]) #converted to column vector
            else:
                for z in xrange(0,k):
                    currin[z] = x[k-1-z]

            ### calculate the predictor output at time t=k
            s[k] = matrixmultiply(transpose(a), currin)[0]

            ### update coeff. for next iteration
            ###    where error err[k] = x[k]- s[k]
            err = x[k]- s[k]

            #intermediate terms used for debugging
            Pc = matrixmultiply(P, currin)
            term = Pc / add(alpha, matrixmultiply(transpose(currin), Pc))
            a = add(a, term * err)

        ###predict one value beyond observed values in x
        k = len(x)
        # current input vector 
        # get M elements, ordered newest to oldest
        currin = zeros((M,1), Float)
        if((k-M) >= 0):   #k-M is an element index 0, 1, ...
            currin=self.flipud(x[k-M:k]) #converted to column vector
        else:
            for z in xrange(0,k):
                currin[z] = x[k-1-z]
    
        # calculate the predictor output at time t=k
        new_prediction = matrixmultiply(transpose(a), currin)[0]

        ### convert to expected format as 0 | 1
        if new_prediction >= 0.5:  #_slightly_ favors 1 over 0, i.e. avl over not
            prediction = 1
        else:
            prediction = 0

        return prediction

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max (=k, =M):  ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print "Alpha:                 ", self.alpha
        print "M:                     ", self.M
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print "Current Uptime Signal (k-bits): \n", self.history
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

########################################
# PredictorHybrid class
#
class PredictorHybrid (Predictor):
    """Predictor that predicts, using a tournament of subpredictors, tracking 
the scores of each subpredictor with a saturation counter.  
Mickens [4] Section 4.1 experimental setup."""

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, tournamentX=2, X=2, d=1, alpha=0.99):
        super(PredictorHybrid, self).__init__(lookahead_max, history_max, prediction_init)

        self.lookahead_max = lookahead_max
        self.history_max = history_max
        self.prediction_init = prediction_init
        self.tournamentX = tournamentX       #max saturation for tournament sat counters
        self.X = X                           #max saturation for subpredictor SatX counters
        self.d = d                           #max differing bits for subpredictor TwidHist
        self.alpha = alpha                   #forgetting factor for linear predictor

        self.tournament = None      #a tournament of counters
        self.subpredictors = []     #indexed list of subpredictors in hybrid
        self.lookahead_queue = []   #{'time':#, 'tournament': <tournmt>, 'predictions':[...]}

        #for saving pickled state
        self.current_time = 0

        ###state may be held in the various subpredictors
        #  -create subpredictor(s)
        #  -subpredictors do initialize their predictions following creation
        print "HYBRID: building simple predictors..."
        predNow = PredictorRightNow(lookahead_max, prediction_init)
        print "HYBRID:      ...predNow done"
        predSatX = PredictorSatX(lookahead_max, history_max, prediction_init, self.X)
        print "HYBRID:      ...predSat"+str(self.X)+" done"

        #saturating counter satX, where X=maximum counter magnitude
        #k = history_max
        [k6, k24, k48, k56] = [6, 24, 48, 56] 
        print "HYBRID: building history predictors..."
        predHist6  = PredictorHistory(lookahead_max, k6,  prediction_init, self.X)
        print "HYBRID:      ...predHist6 done"
        """
        predHist24 = PredictorHistory(lookahead_max, k24, prediction_init, self.X)
        print "HYBRID:      ...predHist24 done"
        predHist48 = PredictorHistory(lookahead_max, k48, prediction_init, self.X)
        print "HYBRID:      ...predHist48 done"
        predHist56 = PredictorHistory(lookahead_max, k56, prediction_init, self.X)
        print "HYBRID:      ...predHist56 done"
        """

        #saturating counter satX, where X=2
        #similar sequences defined as differing by at most d-bits
        #k = history_max
        [k6, k24, k48, k56] = [6, 24, 48, 56] 
        print "HYBRID: building twiddled history predictors..."
        predTwidHist6  = PredictorTwiddledHistory(lookahead_max, k6,  prediction_init, self.X, self.d)
        print "HYBRID:      ...predTwidHist6 done"
        """
        predTwidHist24 = PredictorTwiddledHistory(lookahead_max, k24, prediction_init, self.X, self.d)
        print "HYBRID:      ...predTwidHist6 done"
        predTwidHist48 = PredictorTwiddledHistory(lookahead_max, k48, prediction_init, self.X, self.d)
        print "HYBRID:      ...predTwidHist48 done"
        predTwidHist56 = PredictorTwiddledHistory(lookahead_max, k56, prediction_init, self.X, self.d)
        print "HYBRID:      ...predTwidHist56 done"
        """

        #adaptive linear predictor
        #k = history_max
        [k168, k336] = [168, 336] 
        print "HYBRID: building linear predictors..."
        predLinear168 = PredictorLinear(lookahead_max, k168, prediction_init, self.alpha)
        print "HYBRID:      ...predLinear168 done"
        predLinear336 = PredictorLinear(lookahead_max, k336, prediction_init, self.alpha)
        print "HYBRID:      ...predLinear336 done"

        ### add subpredictors to list
        self.subpredictors.append(predNow)         #0
        self.subpredictors.append(predSatX)        #1
        self.subpredictors.append(predHist6)       #2
        self.subpredictors.append(predTwidHist6)   #3
        self.subpredictors.append(predLinear168)   #4
        self.subpredictors.append(predLinear336)   #5

        """
        ### add subpredictors to list
        self.subpredictors.append(predNow)         #0
        self.subpredictors.append(predHist6)       #2
        self.subpredictors.append(predHist24)      #3
        self.subpredictors.append(predHist48)      #4
        self.subpredictors.append(predHist56)      #5
        self.subpredictors.append(predTwidHist6)   #6
        self.subpredictors.append(predTwidHist24)  #7
        self.subpredictors.append(predTwidHist48)  #8
        self.subpredictors.append(predTwidHist56)  #9
        self.subpredictors.append(predLinear168)   #11
        self.subpredictors.append(predLinear336)   #12

        """

        ###create a tournament of sat counters
        self.tournament = Tournament("SatX", self.tournamentX, self.subpredictors)

        ###for each lookahead period, 
        ###create a set of initial subpredictor predictions
        self.lookahead_queue = []
        for n in xrange(1,lookahead_max+1):
            predictions = []  #one per subpredictor, indexed
            for sub in self.subpredictors:
                predictions.append(prediction_init)
            queue_item = {'time':n, 'predictions':predictions}
            self.lookahead_queue.append(queue_item)

    def update(self, current_time, current_value):

        self.current_time = current_time

        ###update all tournament counters and subpredictors
        self.tournament.update(current_time, current_value)
        for sub in self.subpredictors:
            sub.update(current_time, current_value)

        ###regenerate all predictions to fill queue.
        #  - queued item contains an ordered set of predictions, 
        #    one prediction per subpredictor
        self.lookahead_queue = []
        for n in xrange(current_time+1,current_time+1+self.lookahead_max):
            prediction_set = []  #one per subpredictor, indexed

            for sub in self.subpredictors:
                prediction_set.append(sub.makePrediction(n))
            queue_item = {'time':n, 'predictions':prediction_set}
            self.lookahead_queue.append(queue_item)

    def makePrediction(self, time):
        """Return a prediction for the specified time."""
        #make a prediction for the specified time by retrieving from queue.
        #assume predictions inserted in order of time => sorted Q, no duplicates
        print "predicting for time="+str(time)+" ..."

        ###get prediction set for the specified time
        prediction_set = None
        for pset in self.lookahead_queue:
            #if pset['time'] == time:    #predictions for *exactly* time or none
            if self.isTimeMatch(pset['time'], time):
                prediction_set = pset['predictions']
            elif pset['time'] > time:
                break

        ###use tournament of counters to choose best prediction of the set
        winner = self.tournament.getWinningPredictor()
        prediction = None
        if winner != None:
            prediction = prediction_set[winner.getId()]

        return prediction

    def printLookaheadQueue(self):
        """Print lookahead queue of prediction sets for debugging purposes."""
        for i in xrange(0, len(self.lookahead_queue)):
            print self.lookahead_queue[i]

    def printTournament(self):
        """Print tournament of counters for debugging purposes."""
        self.tournament.printTournament()

    def printWinner(self):
        """Print winning prediction for debugging purposes."""
        if self.tournament.getWinningPredictor() != None:
            print "Winning predictor: " + str(self.tournament.getWinningPredictor().getId())
        else:
            print "Winning predictor: None"

    def __str__(self):
        print "PREDICTOR TYPE:  ", self.__class__.__name__
        print "GENERAL REFERENCE VARIABLES: "
        print "Initalized Prediction: ", self.prediction_init
        print "History Max (=k):       ", self.history_count
        print "Lookahead Period:      ", self.lookahead_count
        print ""
        print "GENERAL STATE VARIABLES: "
        print "Current Time:          ", self.current_time
        print ""
        print "PREDICTOR-SPECIFIC REFERENCE VARIABLES: "
        print ""
        print "TODO:....................???"
        print ""
        print "PREDICTOR-SPECIFIC STATE VARIABLES: "
        print ""
        print "TODO:....................???"
        print ""
        print "CURRENT PREDICTIONS: "
        print self.predictions
        print ""

class Tournament:
    """Tournament of counters
Mickens [4] Section 4.1 experimental setup."""

    ######## PUBLIC METHODS ########

    def __init__(self, counter_type, tournamentX, predictors):
        self.counter_type = counter_type
        self.tournamentX = tournamentX
        self.predictors = predictors

        ###create tournament tree-structure
        self.root_node = self.createTournament()

    def createTournament(self):

        ###TODO:  ......for now this is hand-crafted.  try to change this.
        node0  = TournamentNode(self.tournamentX, 0)  #root node
        node1  = TournamentNode(self.tournamentX, 1)
        node2  = TournamentNode(self.tournamentX, 2)
        node3  = TournamentNode(self.tournamentX, 3)
        node4  = TournamentNode(self.tournamentX, 4)

        """
        node0  = TournamentNode(self.tournamentX, 0)  #root node
        node1  = TournamentNode(self.tournamentX, 1)
        node2  = TournamentNode(self.tournamentX, 2)
        node3  = TournamentNode(self.tournamentX, 3)
        node4  = TournamentNode(self.tournamentX, 4)
        node5  = TournamentNode(self.tournamentX, 5)
        node6  = TournamentNode(self.tournamentX, 6)
        node7  = TournamentNode(self.tournamentX, 7)
        node8  = TournamentNode(self.tournamentX, 8)
        node9  = TournamentNode(self.tournamentX, 9)
        node10 = TournamentNode(self.tournamentX, 10)
        """

        ## LEVEL=0
        node0.setNegative(node1)
        node0.setPositive(node2)

        ## LEVEL=1
        node1.setNegative(node3)
        node1.setPositive(node4)
        node2.setNegativeSubpredictor(SubpredictorNode(4, "Linear",   self.predictors[4]))  #predLinear168
        node2.setPositiveSubpredictor(SubpredictorNode(5, "Linear",   self.predictors[5]))  #predLinear336

        ## LEVEL=2
        node3.setNegativeSubpredictor(SubpredictorNode(0,  "RightNow", self.predictors[0]))   #RightNow
        node3.setPositiveSubpredictor(SubpredictorNode(1,  "SatX",     self.predictors[1]))   #Sat2
        node4.setNegativeSubpredictor(SubpredictorNode(2, "History",         self.predictors[2]))  #predHistory6
        node4.setPositiveSubpredictor(SubpredictorNode(3, "TwiddledHistory", self.predictors[3]))  #predTwidHistory6
        """
        node3.setNegativeSubpredictor(SubpredictorNode(0,  "RightNow", self.predictors[0]))   #RightNow
        node3.setPositiveSubpredictor(SubpredictorNode(1,  "SatX",     self.predictors[1]))   #Sat2
        node4.setNegative(node5)
        node4.setPositive(node6)
        """

        ## LEVEL=3
        """
        node5.setNegative(node7)
        node5.setPositive(node8)
        node6.setNegative(node9)
        node6.setPositive(node10)
        """

        ## LEVEL=4
        """
        node7.setNegativeSubpredictor(SubpredictorNode(2, "History",   self.predictors[2]))  #predHistory6
        node7.setPositiveSubpredictor(SubpredictorNode(3, "History",   self.predictors[3]))  #predHistory24
        node8.setNegativeSubpredictor(SubpredictorNode(4, "History",   self.predictors[4]))  #predHistory48
        node8.setPositiveSubpredictor(SubpredictorNode(5, "History",   self.predictors[5]))  #predHistory56

        node9.setNegativeSubpredictor( SubpredictorNode(6, "TwiddledHistory",   self.predictors[6]))  #predTwidHistory6
        node9.setPositiveSubpredictor( SubpredictorNode(7, "TwiddledHistory",   self.predictors[7]))  #predTwidHistory24
        node10.setNegativeSubpredictor(SubpredictorNode(8, "TwiddledHistory",   self.predictors[8]))  #predTwidHistory48
        node10.setPositiveSubpredictor(SubpredictorNode(9, "TwiddledHistory",   self.predictors[9]))  #predTwidHistory56
        """

        return node0

    def update(self, current_time, current_value):
    ###update all counters based on which subpredictors' predictions 
    ### agree with current observed 

        #recursive update of tree
        self.root_node.update(current_value, current_time, self.predictors)

    def getWinningPredictor(self):
        return self.root_node.getWinningPredictor()

    def printTournament(self):
        print ""
        print "Tournament Data:"
        print "    counter type = " + self.counter_type
        print "    tournament counter saturation range = ",
        print str(-self.tournamentX),
        print "...",
        print str(self.tournamentX)
        print "    predictors: "
        for i in xrange(0,len(self.predictors)):
            print "        " + str(i) + "    ",
            print type(self.predictors[i])

        indent = ""
        self.root_node.printNode(indent)
        print ""

    ######## SUPPORT METHODS ########

class PredictorHybrid_TEST (Predictor):
    """Predictor that predicts, using a tournament of subpredictors, tracking 
the scores of each subpredictor with a saturation counter."""

    def __init__(self, lookahead_max=0, history_max=0, prediction_init=0, tournamentX=2, X=2, d=1, alpha=0.99):
        super(PredictorHybrid_TEST, self).__init__(lookahead_max, history_max, prediction_init)

        self.lookahead_max = lookahead_max
        self.history_max = history_max
        self.prediction_init = prediction_init
        self.tournamentX = tournamentX       #max saturation for tournament sat counters
        self.X = X                           #max saturation for subpredictor SatX counters
        self.d = d                           #max differing bits for subpredictor TwidHist
        self.alpha = alpha                   #forgetting factor for linear predictor

        self.tournament = None      #a tournament of counters
        self.subpredictors = []     #indexed list of subpredictors in hybrid
        self.lookahead_queue = []   #{'time':#, 'tournament': <tournmt>, 'predictions':[...]}


        ###state may be held in the various subpredictors
        #  -create subpredictor(s)
        #  -subpredictors do initialize their predictions following creation
        predRandom = PredictorRandom(lookahead_max, history_max, prediction_init)
        predOn  = PredictorAlways(lookahead_max, history_max, 1)
        predOff = PredictorAlways(lookahead_max, history_max, 0)
        predNow = PredictorRightNow(lookahead_max, history_max, prediction_init)
        predSatX = PredictorSatX(lookahead_max, history_max, prediction_init, self.X)

        #saturating counter satX, where X=maximum counter magnitude
        predHist = PredictorHistory(lookahead_max, history_max, prediction_init, self.X)

        #saturating counter satX, where X=2
        #similar sequences defined as differing by at most d-bits
        predTwidHist = PredictorTwiddledHistory(lookahead_max, history_max, prediction_init, self.X, self.d)

        #adaptive linear predictor
        predLinear = PredictorLinear(lookahead_max, history_max, prediction_init, self.alpha)

        ### add subpredictors to list
        self.subpredictors.append(predNow)      #0
        self.subpredictors.append(predSatX)     #1
        self.subpredictors.append(predHist)     #2
        self.subpredictors.append(predTwidHist) #3
        self.subpredictors.append(predLinear)   #4


        ###create a tournament of sat counters
        self.tournament = Tournament_TEST("SatX", self.tournamentX, self.subpredictors)

        ###for each lookahead period, 
        ###create a set of initial subpredictor predictions
        self.lookahead_queue = []
        for n in xrange(1,lookahead_max+1):
            predictions = []  #one per subpredictor, indexed
            for sub in self.subpredictors:
                predictions.append(prediction_init)
            queue_item = {'time':n, 'predictions':predictions}
            self.lookahead_queue.append(queue_item)

    def update(self, current_time, current_value):

        self.current_time = current_time

        ###update all tournament counters and subpredictors
        self.tournament.update(current_time, current_value)
        for sub in self.subpredictors:
            sub.update(current_time, current_value)

        ###regenerate all predictions to fill queue.
        #  - queued item contains an ordered set of predictions, 
        #    one prediction per subpredictor
        self.lookahead_queue = []
        for n in xrange(current_time+1,current_time+1+self.lookahead_max):
            prediction_set = []  #one per subpredictor, indexed

            for sub in self.subpredictors:
                prediction_set.append(sub.makePrediction(n))
            queue_item = {'time':n, 'predictions':prediction_set}
            self.lookahead_queue.append(queue_item)

    def makePrediction(self, time):
        """Return a prediction for the specified time."""
        #make a prediction for the specified time by retrieving from queue.
        #assume predictions inserted in order of time => sorted Q, no duplicates

        ###get prediction set for the specified time
        prediction_set = None
        for pset in self.lookahead_queue:
            #if pset['time'] == time:    #predictions for *exactly* time or none
            if self.isTimeMatch(pset['time'], time):
                prediction_set = pset['predictions']
            elif pset['time'] > time:
                break

        ###use tournament of counters to choose best prediction of the set
        winner = self.tournament.getWinningPredictor()
        prediction = None
        if winner != None:
            prediction = prediction_set[winner.getId()]

        return prediction

    def printLookaheadQueue(self):
        """Print lookahead queue of prediction sets for debugging purposes."""
        for i in xrange(0, len(self.lookahead_queue)):
            print self.lookahead_queue[i]

    def printTournament(self):
        """Print tournament of counters for debugging purposes."""
        self.tournament.printTournament()

    def printWinner(self):
        """Print winning prediction for debugging purposes."""
        if self.tournament.getWinningPredictor() != None:
            print "Winning predictor: " + str(self.tournament.getWinningPredictor().getId())
        else:
            print "Winning predictor: None"

class Tournament_TEST:
    """Tournament of counters.  
Mickens [4] Section 3.5 Hybrid Predictor explanation."""

    ######## PUBLIC METHODS ########

    def __init__(self, counter_type, tournamentX, predictors):
        self.counter_type = counter_type
        self.tournamentX = tournamentX
        self.predictors = predictors

        ###create tournament tree-structure
        self.root_node = self.createTournament()

    def createTournament(self):

        ###TODO:  ......for now this is hand-crafted.  try to change this.
        node0 = TournamentNode(self.tournamentX, 0)  #root node
        node1 = TournamentNode(self.tournamentX, 1)
        node2 = TournamentNode(self.tournamentX, 2)
        node3 = TournamentNode(self.tournamentX, 3)

        node0.setNegative(node1)
        node0.setPositiveSubpredictor(SubpredictorNode(4, "Linear", self.predictors[4]))  #predLinear

        node1.setNegative(node2)
        node1.setPositive(node3)

        node2.setNegativeSubpredictor(SubpredictorNode(0, "RightNow", self.predictors[0]))  #predNow
        predictor_node = SubpredictorNode(1, "SatX", self.predictors[1])
        node2.setPositiveSubpredictor(predictor_node)  #predSatX

        predictor_node = SubpredictorNode(2, "History", self.predictors[2])
        node3.setNegativeSubpredictor(predictor_node)  #predHist
        predictor_node = SubpredictorNode(3, "TwiddledHistory", self.predictors[3])
        node3.setPositiveSubpredictor(predictor_node)  #predTwidHist

        return node0

    def update(self, current_time, current_value):
    ###update all counters based on which subpredictors' predictions 
    ### agree with current observed 

        #recursive update of tree
        self.root_node.update(current_value, current_time, self.predictors)

    def getWinningPredictor(self):
        return self.root_node.getWinningPredictor()

    def printTournament(self):
        print ""
        print "Tournament Data:"
        print "    counter type = " + self.counter_type
        print "    tournament counter saturation range = ",
        print str(-self.tournamentX),
        print "...",
        print str(self.tournamentX)
        print "    predictors: "
        for i in xrange(0,len(self.predictors)):
            print "        " + str(i) + "    ",
            print type(self.predictors[i])

        indent = ""
        self.root_node.printNode(indent)
        print ""

    ######## SUPPORT METHODS ########

class TournamentNode:
    """Tournament node information"""

    ######## PUBLIC METHODS ########

    ######## TOURNAMENT METHODS ########

    def __init__(self, X, nid=None):
        self.counter = SaturatingCounter(X)
        self.predictorWinner = None
        self.negNode = None
        self.posNode = None
        self.negSubpredictor = None
        self.posSubpredictor = None
        self.nid = nid

    def setWinningPredictor(self, neg_predictor, pos_predictor):
        if self.getCounterValue() > 0:  #positive
            self.predictorWinner = pos_predictor
        elif self.getCounterValue() < 0:  #negative
            self.predictorWinner = neg_predictor
        else:  #0 => never ever updated since counter creation

            if (globals()['_RANDOM_DISALLOWED']):  ##FOR UNIT TESTING
                ## always pick same side
                self.predictorWinner = neg_predictor  #creates neg-side bias!

            else:  ## NORMAL OPERATIONS

                ##pick randomly to avoid bias
                if randrange(0, 2):  #integer 0 or 1
                    self.predictorWinner = pos_predictor
                else:
                    self.predictorWinner = neg_predictor
            

    def setNegative(self, node):
        self.negNode = node

    def setPositive(self, node):
        self.posNode = node

    def setNegativeSubpredictor(self, predictor):
        self.negSubpredictor = predictor

    def setPositiveSubpredictor(self, predictor):
        self.posSubpredictor = predictor

    def getId(self):
        return self.nid

    def getWinningPredictor(self):
        return self.predictorWinner

    def getNegative(self):
        return self.negNode

    def getPositive(self):
        return self.posNode

    def getNegativeSubpredictor(self):
        return self.negSubpredictor

    def getPositiveSubpredictor(self):
        return self.posSubpredictor

    def increment(self):
        self.counter.increment()

    def decrement(self):
        self.counter.decrement()

    def getCounterValue(self):
        return self.counter.getValue()

    def printNode(self, indent):

        print indent + "Tournament Node Data:"
        print indent + "    nodeId = " + str(self.nid)
        print indent + "    counter value = " + str(self.counter.getValue())
        if self.predictorWinner != None:
            print indent + "    winning predictor index = " + str(self.predictorWinner.getId())
        else:
            print indent + "    winning predictor index = None"
        print indent + ""
        if self.negSubpredictor != None:
            print indent + "    negative predictor index = " + str(self.negSubpredictor.getId())
        else:
            print indent + "    negative predictor index = None"
        if self.posSubpredictor != None:
            print indent + "    positive predictor index = " + str(self.posSubpredictor.getId())
        else:
            print indent + "    positive predictor index = None"
        print indent + ""
        if self.negNode != None:
            print indent + "NEGATIVE SUBNODE:"
            self.negNode.printNode(indent + "    ")
        if self.posNode != None:
            print indent + "POSITIVE SUBNODE:"
            self.posNode.printNode(indent + "    ")

    def update(self, current_value, current_time, predictors):
        ### update all counters bottom-up
        #     use DFS where bottom counter nodes are leaves
        #     with two predictor children

        ###process children first for DFS
        [neg_node, pos_node] = [self.getNegative(), self.getPositive()]
        neg_predictor = self.getNegativeSubpredictor()
        pos_predictor = self.getPositiveSubpredictor()

        #update children nodes, if any
        if neg_node != None:
            neg_predictor = neg_node.update(current_value, current_time, predictors)
        if pos_node != None:
            pos_predictor = pos_node.update(current_value, current_time, predictors)

        ###process start node
        #     compare subpredictions to actual value and update counter
        neg_is_right = self.isPredictorRight(neg_predictor, current_value, current_time)
        pos_is_right = self.isPredictorRight(pos_predictor, current_value, current_time)

        if (neg_is_right) & (not pos_is_right):
            self.decrement()
        elif (pos_is_right) & (not neg_is_right):
            self.increment()
        else:  ###do nothing if both right or both wrong
            pass

        self.setWinningPredictor(neg_predictor, pos_predictor)
        return self.getWinningPredictor()

    ######## SUPPORT METHODS ########

    def isPredictorRight(self, predictor, right_value, current_time):
        is_right = False
        if predictor != None:
            if predictor.makePrediction(current_time) == right_value:
                is_right = True
        return is_right

class SubpredictorNode:
    """Wrapper for a Predictor subclass"""

    def __init__(self, pid, predictor_type, predictor):
        self.pid = pid
        self.predictor_type = predictor_type
        self.predictor = predictor

    def getId(self):
        return self.pid

    def getType(self):
        return self.predictor_type

    def getPredictor(self):
        return self.predictor

    def update(self, current_time, current_value):
        self.predictor.update(current_time, current_value)

    def makePrediction(self, time):
        return self.predictor.makePrediction(time)

