"""
standalone_preprocessor.py

This standalone module processes raw data files and generates a 
pickled EventGenerator that contains all of the DeviceEvents 
and AppEvents for a run of the simulator.

Author: Sami Rollins

"""

import os, email, sys, random, pickle, getopt
from RandomArray import poisson

from simconstants import *
from applayer_utils import *
from device_utils import *

def toBool(val):
    """
    Method to convert a string value (val) to a boolean.
    1 is converted to True and all other values are converted
    to False.  val should be 1, 0, or M (missing data).
    """
    if val == '1':
        return True
    elif val == '0' or val == 'M':
        return False
    else:
        return False
    
def toInt(val):
    """
    Method to convert a string value (val) to an integer.
    Nonnumerical values are converted to 0.
    """
    try:
        returnval = int(val)
        return returnval
    except:
        return 0
    
def getMeasurement(file):
    """
    getMeasurement takes as input a file object and returns
    a single measurement from the file.  Measurements are
    expected to be of the form:
    
    Tue, 01 May 2007 05:34:39 -0000
    battery 100
    onAC 1
    cpu 1
    disk 3466
    connected 1
    idle_time 5749
    
    The method returns a date object and a DeviceStateInfo
    object representing the state of the device at the time of the
    measurement.
    """
    #date
    date = file.readline()
    if not date: return None, None
    date = datetime.datetime.fromtimestamp(email.Utils.mktime_tz(email.Utils.parsedate_tz(date)))
    
    #battery
    bat = file.readline()
    bat = bat.split()[1]
    if not bat: return None, None
    bat = toInt(bat)
                   
    #onAC
    onac = file.readline().split()[1]
    if not onac: return None, None
    onac = toBool(onac)
                   
    #cpu
    cpu = file.readline()
    if not cpu: return None, None
                 
    #disk
    disk = file.readline()
    if not disk: return None, None
                
    #connected
    ison = file.readline().split()[1]
    if not ison: return None, None
    ison = toBool(ison)
                    
    #idle time
    idle = file.readline()
    if not idle: return None, None
    
    return date, DeviceStateInfo(ison, bat, onac)

def stateChange(laststateinfo, thisstateinfo):
    """
    stateChange determines whether there has been a state change 
    between the measurement which produced laststateinfo and the
    measurement that produced thisstateinfo.  The method returns
    a DeviceChangeInfo object detailing the state change.  If
    there is no state change None is returned.
    """
    onchange = False
    batchange = False
    onacchange = False
    
    #if this is the first measurement
    #  all items have changed if the node is on
    if laststateinfo == None and thisstateinfo.ison:
        return DeviceChangeInfo(True, True, True)
    elif laststateinfo == None:
        return None
    
    #did node turn off or on
    if (laststateinfo.ison and not thisstateinfo.ison) or \
        (not laststateinfo.ison and thisstateinfo.ison):
        onchange = True
   
    #did node plug in or unplug
    if (not laststateinfo.onac and thisstateinfo.onac) or \
        (laststateinfo.onac and not thisstateinfo.onac):
        onacchange = True

    #did battery change
    #exclude situations where battery changes but device is not connected
    if thisstateinfo.ison and laststateinfo.bat != thisstateinfo.bat:
        batchange = True

    #if there has been a change, return a new StateChange
    if(onchange == True or batchange == True or onacchange == True):
        return DeviceChangeInfo(onchange, batchange, onacchange)

    return None

def generateAppEvents(id, events, dates, totalonline, allids, freq, rate):
    """
    Generate the AppEvents for a given node.  
    id - id of the node for which the events will be generated
    events - queue where events will be inserted
    dates - list of dates representing on times for the node
    totalonline - total number of measurements where node is online
    allids - list of ids for all devices
    freq - freqency of app-layer events
    rate - rate of data sampling - number of samples per hour
    """
    #generate array with poisson distribution of .5/6
    #each element of resulting array represents the number of
    #msgs generated in that time slot 
    #e.g., RandomArray.poisson(.5/6, 4) => [1, 0, 0, 0] -
    #1 msg at time 0, 0 msgs at time 1, etc
    uuid = 1
    newevs = poisson((freq/rate), totalonline)
    #for each time slot
    for i in range(len(newevs)):
        #for each msg to be sent at that time slot
        for j in range(newevs[i]):
            #generate the destination node
            num = random.randint(0, len(allids)-1)
            #make sure to exclude self
            while id == allids[num]:
                num = random.randint(0, len(allids)-1)

            #generate unique id for event
            uuidstr = id + "-" + str(uuid)
            uuid += 1
                     
            #generate an event                 
            event = AppEvent(id, dates[i], uuidstr, id, allids[num], ORIG, dates[i])
            events.insertEvent(event)
    
    
    
    
def processFiles(dirname, freq, rate):
    """
    Main logic of the program that processes the data files 
    in the directory dirname and returns and EventGenerator
    containing all of the DeviceEvents derived from the 
    data files as well as the AppEvents generated
    by using a poisson distribution.  It also returns a list
    of the ids of all devices.
    freq - freqency of app-layer events
    rate - rate of data sampling - number of samples per hour
    """
    events = EventGenerator()
    allids = []
    finaldate = None
    startdate = None
    for root, dirs, files in os.walk(dirname):
        #create a list of ids for all devices
        #in the system
        #used to generate the App events
        for name in files:
            if name.startswith("."): continue
            allids.append(name.split(".")[0])
        for name in files:                                
            if name.startswith("."): continue
            totalonline = 0      
            firstdate = None
            lastdate = None
            dates = []
            fname = dirname + "/" + name
            id = name.split(".")[0]
            file = open(fname, "r")
            laststateinfo = None
            
            while 1:                
                date, thisstateinfo = getMeasurement(file)
                if firstdate == None:
                    firstdate = date
                #if the device is on, then keep track of the time 
                #of this measurement and add to the total number 
                #of measurements reporting online
                #this will be used to generate the App events later
                if not thisstateinfo: break
                if thisstateinfo.ison:
                    totalonline += 1
                    dates.append(date)

                #determine if there has been a state change
                #if yes, add a new DeviceEvent
                changeinfo = stateChange(laststateinfo, thisstateinfo)
                if(changeinfo):
                    events.insertEvent(DeviceEvent(id, date, thisstateinfo, changeinfo))
                    
                laststateinfo = thisstateinfo
                      
                #save last date for turning off at end of run        
                lastdate = date
                
            #generate AppEvents and insert into EventGenerator
            generateAppEvents(id, events, dates, totalonline, allids, freq, rate)

            if laststateinfo.ison:
                #add one last off event at the end of the run
                events.insertEvent(DeviceEvent(id, (lastdate+datetime.timedelta(seconds=1)), DeviceStateInfo(False, 100, True), DeviceChangeInfo(True, False, False)))
            #save final time for triggering end of simulation
            if finaldate == None or lastdate > finaldate:
                finaldate = lastdate
            if startdate == None or firstdate < startdate:
                startdate = firstdate
        return events, allids, startdate, finaldate

def pickleEventGenerator(events, allids, startdate, finaldate, filename):
    """
    dump the EventGenerator, the list of ids, the startdate, and the final event time to a file
    """
    f = open(filename, "w")
    pickle.dump(events, f)
    pickle.dump(allids, f)
    pickle.dump(startdate, f)
    pickle.dump(finaldate, f)
    f.close()


def usage():
    sys.stderr.write("\n")
    sys.stderr.write("Usage: standalone_preprocessor.py -i <input_directory> -o <output_file> -f  <app_event_frequency (per hour)> -r <data_points_per_hour\n")
    sys.stderr.write("\n")
    sys.stderr.write("_______________________\n")
    sys.stderr.write("_______________________\n")
    sys.stderr.write("\n")
    sys.stderr.write("       <input_directory>     directory containing raw input data\n")
    sys.stderr.write("                             in the form of consolidated text files\n")
    sys.stderr.write("\n")
    sys.stderr.write("       <output_file>         absolute path and file name were pickle should be created\n")
    sys.stderr.write("\n")
    sys.stderr.write("       <app_event_frequency>         number of app layer events to generate per hour\n")
    sys.stderr.write("\n")
    sys.stderr.write("       <data_points_per_hour>         number of data points per hour (e.g., 3 if data is 20 min)\n")
    sys.stderr.write("\n")

def main(argv=None):
    
    argv = sys.argv
    
    try:                                
        opts, args = getopt.getopt(argv[1:], "i:o:f:r:")
        if len(opts) < 4:
            usage()
            sys.exit(2)
        for o, a, in opts:
            if o == "-i":
                indir = a
            elif o == "-o":
                outfile = a
            elif o == "-f":
                freq = float(a)
            elif o == "-r":
                rate = float(a)
    except:
        usage()
        sys.exit(2)

    events, allids, startdate, finaldate = processFiles(indir, freq, rate)
    pickleEventGenerator(events, allids, startdate, finaldate, outfile)
    
    """
    f = open(outfile, "r")
    print "loading..."
    loadedevents = pickle.load(f)
    f.close()
    
    while loadedevents.hasNext():
        print loadedevents.getNextEvent()
    """
if (__name__ == "__main__"):
    sys.exit(main())